<?php

declare(strict_types=1);

namespace Drupal\taxonomy_term_replacement\Form;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for replacing one taxonomy term with another.
 */
class ReplaceTermForm extends FormBase {

  /**
   * The entity type manager.
   *
   * @var EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  private $entityTypeBundleInfo;

  /**
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  private $entityFieldManager;

  /**
   * Constructs a form.
   *
   * @param EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info, EntityFieldManagerInterface $entityFieldManager) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->entityFieldManager = $entityFieldManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_field.manager')
    );
  }

  public function getFormId() {
    return 'taxonomy_term_replacement_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state) {
    // Load all vocabularies and terms to allow selects.
    $vocabularies = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple();
    $vocab_options = [];
    foreach ($vocabularies as $vid => $vocab) {
      $vocab_options[$vid] = $vocab->label();
    }

    $form['vocabulary'] = [
      '#type' => 'select',
      '#title' => $this->t('Vocabulary'),
      '#options' => $vocab_options,
      '#empty_option' => $this->t('- Select vocabulary -'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateTermSelects',
        'event' => 'change',
        'wrapper' => 'term-select-wrapper',
      ],
    ];

    $form['term_selects'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'term-select-wrapper'],
    ];

    $vocabulary_selected = $form_state->getValue('vocabulary');
    $terms_options = [];
    if ($vocabulary_selected) {
      $terms = $this->entityTypeManager->getStorage('taxonomy_term')
        ->loadTree($vocabulary_selected);
      foreach ($terms as $term) {
        // tree gives name, depth, tid
        $terms_options[$term->tid] = str_repeat('-', $term->depth) . ' ' . $term->name;
      }
    }

    $form['term_selects']['old_term'] = [
      '#type' => 'select',
      '#title' => $this->t('Term to replace'),
      '#options' => $terms_options,
      '#empty_option' => $this->t('- Select term -'),
      '#required' => TRUE,
    ];

    $form['term_selects']['new_term'] = [
      '#type' => 'select',
      '#title' => $this->t('Replacement term'),
      '#options' => $terms_options,
      '#empty_option' => $this->t('- Select term -'),
      '#required' => TRUE,
    ];

    $form['delete_old'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Delete old term after replacement'),
      '#description' => $this->t('If checked, the old term will be deleted after updating all references.'),
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Replace'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $old_tid = $form_state->getValue(['term_selects', 'old_term']);
    $new_tid = $form_state->getValue(['term_selects', 'new_term']);
    if ($old_tid && $new_tid && $old_tid == $new_tid) {
      $form_state->setErrorByName('new_term', $this->t('Replacement term must be different from the term being replaced.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $old_tid = $form_state->getValue('old_term');
    $new_tid = $form_state->getValue('new_term');
    $delete_old = $form_state->getValue('delete_old');
    $vocabulary = $form_state->getValue('vocabulary');

    // Find all entity reference fields that reference taxonomy_term of this vocabulary.
    // For simplicity, here we assume content type nodes; you may want to generalize.

    // Get all node bundles.
    $bundles = array_keys($this->entityTypeBundleInfo->getBundleInfo('node'));
    $fields_to_update = [];

    foreach ($bundles as $bundle) {
      $field_definitions = $this->entityFieldManager->getFieldDefinitions('node', $bundle);
      foreach ($field_definitions as $field_name => $field_def) {
        if ($field_def->getType() === 'entity_reference' && $field_def->getSetting('target_type') === 'taxonomy_term') {
          $handler_settings = $field_def->getSetting('handler_settings') ?? [];
          if (!empty($handler_settings['target_bundles']) && in_array($vocabulary, $handler_settings['target_bundles'])) {
            $fields_to_update[$bundle][] = $field_name;
          }
        }
      }
    }

    // Now update all nodes having the old term.
    $node_storage = $this->entityTypeManager->getStorage('node');
    foreach ($fields_to_update as $bundle => $field_names) {
      $query = $this->entityTypeManager->getStorage('node')->getQuery()
        ->condition('type', $bundle)
        ->accessCheck(FALSE);
      $nids = $query->execute();
      if (!empty($nids)) {
        $nodes = $node_storage->loadMultiple($nids);
        foreach ($nodes as $node) {
          $changed = FALSE;
          foreach ($field_names as $field_name) {
            if ($node->hasField($field_name)) {
              $values = $node->get($field_name)->getValue();
              // Handle single or multiple values
              foreach ($values as &$value) {
                if ($value['target_id'] == $old_tid) {
                  $value['target_id'] = $new_tid;
                  $changed = TRUE;
                }
              }
              if ($changed) {
                $node->set($field_name, $values);
              }
            }
          }
          if ($changed) {
            $node->save();
          }
        }
      }
    }

    $this->messenger()->addStatus($this->t('Replaced term %old with %new in all content. %delete_msg', [
      '%old' => Term::load($old_tid)->label(),
      '%new' => Term::load($new_tid)->label(),
      '%delete_msg' => $delete_old ? $this->t('Old term deleted.') : $this->t('Old term kept.'),
    ]));

    // Optionally delete old term.
    if ($delete_old) {
      $old_term = Term::load($old_tid);
      if ($old_term) {
        $old_term->delete();
      }
    }
  }

  /**
   * AJAX callback to update term selects when vocabulary changes.
   */
  public function updateTermSelects(array &$form, FormStateInterface $form_state) {
    return $form['term_selects'];
  }
}