# Taxonomy Usage Module

A Drupal module that provides comprehensive taxonomy term usage detection and prevents deletion of terms that are still in use.

## Features

- **Comprehensive Usage Detection**: Finds taxonomy term usage across all entity types and fields
- **Draft Content Support**: Detects usage in both published and draft content
- **Deletion Prevention**: Prevents accidental deletion of taxonomy terms that are still in use
- **User-Friendly Display**: Provides detailed usage information with links to content
- **Performance Optimized**: Uses efficient database queries and caching
- **Flexible Configuration**: Configurable entity types, result limits, and draft checking

## Entity Type Support

### Core Entity Types (Always Supported)
- **Nodes**: Full support including draft revisions
- **Media**: Complete support for media entities
- **Users**: User entity taxonomy references
- **Comments**: Comment entity support

### Optional Entity Types (Conditional Support)
- **Paragraphs**: Full support when Paragraphs module is enabled
  - Automatic parent entity linking
  - Draft paragraph detection
  - Nested paragraph hierarchy traversal
  - Graceful degradation when module not available

## Paragraphs Module Compatibility

The module is designed to work seamlessly with or without the Paragraphs module:

### When Paragraphs Module is Available
- Full paragraph entity support
- Parent entity linking (paragraphs link to their parent nodes/media)
- Draft paragraph detection through parent entity status
- Nested paragraph hierarchy traversal
- Database queries include paragraph-specific tables

### When Paragraphs Module is Not Available
- Module functions normally with core entity types
- Paragraph-specific code is safely bypassed
- No errors or warnings generated
- Performance remains optimal

### Implementation Details
The module uses conditional checks throughout:
```php
if ($entity->getEntityTypeId() === 'paragraph' && $this->isParagraphsModuleAvailable()) {
  // Paragraph-specific functionality
}
```

## Configuration

### Default Configuration
```yaml
prevent_deletion: true
check_drafts: true
max_results: 50
entity_types:
  node: node
```

### Adding Paragraphs Support
To enable paragraphs support (when module is available):
```yaml
entity_types:
  node: node
  paragraph: paragraph
  media: media
```

## Installation

1. Place the module in your `modules/custom` directory
2. Enable the module: `drush en taxonomy_usage`
3. Configure entity types at `/admin/config/taxonomy/usage`

## Usage

### Preventing Term Deletion
The module automatically prevents deletion of terms in use. Users will see a detailed usage report instead of the delete form.

### Manual Usage Checking
```php
$usage_detector = \Drupal::service('taxonomy_usage.detector');
$is_in_use = $usage_detector->isTermInUse($term_id);
$usage_data = $usage_detector->getTermUsageByField($term_id);
```

### Display Usage Information
```php
$usage_display = \Drupal::service('taxonomy_usage.display');
$html = $usage_display->renderUsageTable($term_id);
```

## Architecture

### Services
- `taxonomy_usage.detector`: Core usage detection logic
- `taxonomy_usage.display`: Usage information rendering
- `taxonomy_usage.validator`: Term deletion validation

### Key Classes
- `TaxonomyUsageDetector`: Main detection engine
- `TaxonomyUsageDisplay`: HTML rendering and formatting
- `TaxonomyUsageValidator`: Deletion prevention logic

### Database Optimization
- Efficient field discovery using entity field manager
- Optimized queries with proper indexing
- Result caching to improve performance
- Batch processing for large datasets

## Compatibility

### Drupal Version
- Drupal 9.x, 10.x, and 11.x compatible

### Module Dependencies
- **Required**: Core Taxonomy module
- **Optional**: Paragraphs module (enhanced functionality when available)
- **Compatible**: Works with any entity type that supports taxonomy fields

### Performance Considerations
- Caches usage data for 5 minutes
- Limits results to prevent memory issues
- Uses database queries instead of entity loading where possible
- Graceful handling of large datasets

## Troubleshooting

### Common Issues

**Issue**: Paragraphs not showing in usage
**Solution**: Ensure paragraphs are enabled in configuration and the Paragraphs module is installed

**Issue**: Performance problems with large sites
**Solution**: Reduce `max_results` in configuration or disable draft checking

**Issue**: Missing usage data
**Solution**: Clear caches and verify entity types are enabled in configuration

### Debug Information
Enable verbose logging to see detailed usage detection:
```php
$usage_data = $usage_detector->getTermUsageByField($term_id);
\Drupal::logger('taxonomy_usage')->info('Usage data: @data', ['@data' => print_r($usage_data, TRUE)]);
```

## Development

### Adding New Entity Type Support
1. Add entity type to configuration schema
2. Update default configuration
3. Test with the new entity type
4. Add any entity-specific handling if needed

### Extending Functionality
The module uses a service-based architecture that makes it easy to extend:
- Override services in your custom module
- Add new display formatters
- Implement custom usage detection logic

## Credits

This module was created with the assistance of AI technology to accelerate development and ensure comprehensive documentation and code quality.

## License

This module is licensed under the GPL v2 or later.
