<?php

namespace Drupal\taxonomy_usage\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\taxonomy\TermInterface;
use Drupal\taxonomy_usage\TaxonomyUsageDisplay;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for taxonomy usage pages.
 */
class TaxonomyUsageController extends ControllerBase {

  /**
   * The taxonomy usage display service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDisplay
   */
  protected $usageDisplay;

  /**
   * Constructs a TaxonomyUsageController object.
   *
   * @param \Drupal\taxonomy_usage\TaxonomyUsageDisplay $usage_display
   *   The taxonomy usage display service.
   */
  public function __construct(TaxonomyUsageDisplay $usage_display) {
    $this->usageDisplay = $usage_display;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('taxonomy_usage.display')
    );
  }

  /**
   * Display taxonomy term usage page.
   *
   * @param \Drupal\taxonomy\TermInterface $taxonomy_term
   *   The taxonomy term entity.
   *
   * @return array
   *   A render array for the usage page.
   */
  public function usagePage(TermInterface $taxonomy_term) {
    $build = [];

    $build['#title'] = $this->t('Usage for "@term"', [
      '@term' => $taxonomy_term->getName(),
    ]);

    $build['term_info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="taxonomy-term-info"><p><strong>' . $this->t('Term:') . '</strong> ' . $taxonomy_term->getName() . ' (' . $this->t('ID: @id', ['@id' => $taxonomy_term->id()]) . ')</p><p><strong>' . $this->t('Vocabulary:') . '</strong> ' . $taxonomy_term->bundle() . '</p></div>',
      '#weight' => -10,
    ];

    $build['usage_display'] = [
      '#type' => 'markup',
      '#markup' => $this->usageDisplay->renderUsageTable($taxonomy_term->id()),
      '#weight' => 0,
    ];

    $edit_url = $taxonomy_term->toUrl('edit-form')->toString();
    $delete_url = $taxonomy_term->toUrl('delete-form')->toString();
    $edit_text = $this->t('Edit term');
    $delete_text = $this->t('Delete term');
    $markup = '<div class="taxonomy-usage-actions"><p>';
    $markup .= '<a href="' . $edit_url . '">' . $edit_text . '</a> | ';
    $markup .= '<a href="' . $delete_url . '">' . $delete_text . '</a>';
    $markup .= '</p></div>';

    $build['actions'] = [
      '#type' => 'markup',
      '#markup' => $markup,
      '#weight' => 10,
    ];

    return $build;
  }

}
