<?php

namespace Drupal\taxonomy_usage\Form;

use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a deletion confirmation form for taxonomy terms with usage checking.
 */
class TaxonomyTermDeleteForm extends ContentEntityDeleteForm {

  /**
   * The taxonomy usage display service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDisplay
   */
  protected $usageDisplay;

  /**
   * The taxonomy usage validator service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageValidator
   */
  protected $usageValidator;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->usageDisplay = $container->get('taxonomy_usage.display');
    $instance->usageValidator = $container->get('taxonomy_usage.validator');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildForm($form, $form_state);

    $term = $this->getEntity();

    // Add usage information.
    $usage_html = $this->usageDisplay->renderUsageTable($term->id());

    $form['taxonomy_usage'] = [
      '#type' => 'markup',
      '#markup' => $usage_html,
      '#weight' => -10,
    ];

    // Add CSS for styling.
    $form['#attached']['library'][] = 'taxonomy_usage/usage_display';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $term = $this->getEntity();
    $validation = $this->usageValidator->validateTermDeletion($term->id());

    if (!$validation['valid']) {
      $form_state->setError($form, $validation['message']);
    }
  }

}
