<?php

namespace Drupal\taxonomy_usage\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for taxonomy usage settings.
 */
class TaxonomyUsageSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['taxonomy_usage.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'taxonomy_usage_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('taxonomy_usage.settings');

    $form['prevent_deletion'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Prevent deletion of terms in use'),
      '#description' => $this->t('When enabled, taxonomy terms that are referenced by content cannot be deleted.'),
      '#default_value' => $config->get('prevent_deletion') ?? TRUE,
    ];

    $form['check_drafts'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Check draft revisions'),
      '#description' => $this->t('When enabled, the module will also check draft revisions for taxonomy term usage.'),
      '#default_value' => $config->get('check_drafts') ?? TRUE,
    ];

    $form['max_results'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum results to display'),
      '#description' => $this->t('Maximum number of entities to show in usage reports. Set to 0 for unlimited.'),
      '#default_value' => $config->get('max_results') ?? 50,
      '#min' => 0,
    ];

    $form['entity_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Entity types to check'),
      '#description' => $this->t('Select which entity types should be checked for taxonomy term usage.'),
      '#options' => $this->getEntityTypeOptions(),
      '#default_value' => $config->get('entity_types') ?? ['node'],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('taxonomy_usage.settings')
      ->set('prevent_deletion', $form_state->getValue('prevent_deletion'))
      ->set('check_drafts', $form_state->getValue('check_drafts'))
      ->set('max_results', $form_state->getValue('max_results'))
      ->set('entity_types', array_filter($form_state->getValue('entity_types')))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available entity type options.
   *
   * @return array
   *   Array of entity type options.
   */
  protected function getEntityTypeOptions() {
    $entity_types = \Drupal::entityTypeManager()->getDefinitions();
    $options = [];

    foreach ($entity_types as $entity_type_id => $entity_type) {
      // Only include content entity types that can have fields.
      if ($entity_type->entityClassImplements('\Drupal\Core\Entity\ContentEntityInterface')) {
        $options[$entity_type_id] = $entity_type->getLabel();
      }
    }

    return $options;
  }

}
