<?php

namespace Drupal\taxonomy_usage;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\taxonomy\Entity\Term;

/**
 * Service for detecting taxonomy term usage across all fields and entity types.
 */
class TaxonomyUsageDetector {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Static cache for taxonomy fields.
   *
   * @var array
   */
  protected static $taxonomyFields;

  /**
   * Check if the paragraphs module is available.
   *
   * @return bool
   *   TRUE if paragraphs module is enabled, FALSE otherwise.
   */
  protected function isParagraphsModuleAvailable() {
    return $this->moduleHandler->moduleExists('paragraphs');
  }

  /**
   * Check if a database table exists.
   *
   * @param string $table_name
   *   The table name to check.
   *
   * @return bool
   *   TRUE if table exists, FALSE otherwise.
   */
  protected function tableExists($table_name) {
    return $this->database->schema()->tableExists($table_name);
  }

  /**
   * Get the safe table name for a field.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $field_name
   *   The field name.
   * @param bool $revision
   *   Whether to get the revision table name.
   *
   * @return string
   *   The table name.
   */
  protected function getSafeTableName($entity_type, $field_name, $revision = FALSE) {
    if ($revision) {
      return "{$entity_type}_revision__{$field_name}";
    }
    return "{$entity_type}__{$field_name}";
  }

  /**
   * Constructs a TaxonomyUsageDetector object.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, Connection $database, EntityFieldManagerInterface $entity_field_manager, CacheBackendInterface $cache, ConfigFactoryInterface $config_factory, ModuleHandlerInterface $module_handler) {
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
    $this->entityFieldManager = $entity_field_manager;
    $this->cache = $cache;
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Get comprehensive taxonomy term usage across all fields and vocabularies.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return array
   *   Structured array of usage data grouped by field.
   */
  public function getTermUsageByField($tid) {
    $term = Term::load($tid);
    if (!$term) {
      return [];
    }

    // Check cache first.
    $cache_key = "taxonomy_usage:term:{$tid}";
    $cached = $this->cache->get($cache_key);
    if ($cached && $cached->data) {
      return $cached->data;
    }

    $config = $this->configFactory->get('taxonomy_usage.settings');
    $max_results = $config->get('max_results') ?? 50;
    $check_drafts = $config->get('check_drafts') ?? TRUE;
    $enabled_entity_types = $config->get('entity_types') ?? ['node'];

    $usage_data = [];

    // Get taxonomy fields for this vocabulary only.
    $taxonomy_fields = $this->getTaxonomyFieldsForVocabulary($term->bundle(), $enabled_entity_types);

    // Group fields by entity type to ensure representative sampling.
    $fields_by_entity_type = [];
    foreach ($taxonomy_fields as $field_info) {
      $entity_type = $field_info['entity_type'];
      $fields_by_entity_type[$entity_type][] = $field_info;
    }

    // Calculate per-entity-type limit to ensure fair distribution.
    $entity_type_count = count($fields_by_entity_type);
    $per_entity_type_limit = $entity_type_count > 0 ? max(1, intval($max_results / $entity_type_count)) : $max_results;

    foreach ($fields_by_entity_type as $entity_type => $entity_fields) {
      $entity_type_found = 0;
      $per_field_limit = count($entity_fields) > 0 ? max(1, intval($per_entity_type_limit / count($entity_fields))) : $per_entity_type_limit;

      foreach ($entity_fields as $field_info) {
        if ($max_results > 0 && $entity_type_found >= $per_entity_type_limit) {
          break;
        }

        $field_name = $field_info['field_name'];

        // Use direct database query for better performance.
        $entities = $this->findEntitiesUsingTermOptimized($entity_type, $field_name, $tid, $per_field_limit, $check_drafts);

        if (!empty($entities)) {
          $field_key = $entity_type . '.' . $field_info['bundle'] . '.' . $field_name;
          $usage_data[$field_key] = [
            'field_name' => $field_name,
            'field_label' => $field_info['label'],
            'entity_type' => $entity_type,
            'bundle' => $field_info['bundle'],
            'bundle_label' => $field_info['bundle_label'],
            'entities' => $entities,
          ];
          $entity_type_found += count($entities);
        }
      }
    }

    // Cache the results for 5 minutes.
    $this->cache->set($cache_key, $usage_data, time() + 300, ['taxonomy_term:' . $tid]);

    return $usage_data;
  }

  /**
   * Get taxonomy fields for a specific vocabulary (optimized).
   *
   * @param string $vocabulary
   *   The vocabulary machine name.
   * @param array $entity_types
   *   Enabled entity types to check.
   *
   * @return array
   *   Array of field information.
   */
  protected function getTaxonomyFieldsForVocabulary($vocabulary, array $entity_types) {
    // Use static cache to avoid repeated queries.
    $cache_key = $vocabulary . ':' . implode(',', $entity_types);
    if (isset(static::$taxonomyFields[$cache_key])) {
      return static::$taxonomyFields[$cache_key];
    }

    $fields = [];

    // Only get field configs that target our vocabulary.
    $field_configs = $this->entityTypeManager
      ->getStorage('field_config')
      ->loadByProperties([
        'field_type' => 'entity_reference',
      ]);

    foreach ($field_configs as $field_config) {
      $entity_type = $field_config->getTargetEntityTypeId();

      // Skip if entity type not enabled.
      if (!in_array($entity_type, $entity_types)) {
        continue;
      }

      $settings = $field_config->getSettings();
      if ($settings['target_type'] !== 'taxonomy_term') {
        continue;
      }

      $handler_settings = $settings['handler_settings'] ?? [];
      $target_bundles = array_keys($handler_settings['target_bundles'] ?? []);

      // Skip if this field doesn't target our vocabulary.
      if (!in_array($vocabulary, $target_bundles)) {
        continue;
      }

      // Get bundle label efficiently.
      $bundle_label = $field_config->getTargetBundle();
      try {
        $bundle_info = $this->entityTypeManager->getDefinition($entity_type)->getBundleEntityType();
        if ($bundle_info) {
          $bundle_entity = $this->entityTypeManager->getStorage($bundle_info)->load($field_config->getTargetBundle());
          if ($bundle_entity) {
            $bundle_label = $bundle_entity->label();
          }
        }
      }
      catch (\Exception $e) {
        // Use machine name if bundle label lookup fails.
      }

      $fields[] = [
        'field_name' => $field_config->getName(),
        'entity_type' => $entity_type,
        'bundle' => $field_config->getTargetBundle(),
        'bundle_label' => $bundle_label,
        'label' => $field_config->getLabel(),
        'target_bundles' => $target_bundles,
      ];
    }

    static::$taxonomyFields[$cache_key] = $fields;
    return $fields;
  }

  /**
   * Find entities using a term with optimized database queries.
   *
   * @param string $entity_type
   *   The entity type.
   * @param string $field_name
   *   The field name.
   * @param int $tid
   *   The taxonomy term ID.
   * @param int $limit
   *   Maximum number of results to return.
   * @param bool $check_drafts
   *   Whether to check draft revisions.
   *
   * @return array
   *   Array of entity usage data.
   */
  protected function findEntitiesUsingTermOptimized($entity_type, $field_name, $tid, $limit, $check_drafts) {
    $entities = [];

    // Use direct database query for better performance.
    $data_table = "{$entity_type}__{$field_name}";
    if (!$this->database->schema()->tableExists($data_table)) {
      return [];
    }

    // Query current data table.
    $query = $this->database->select($data_table, 'f');
    $query->fields('f', ['entity_id', 'revision_id']);
    $query->condition("f.{$field_name}_target_id", $tid);
    $query->range(0, $limit);

    $results = $query->execute()->fetchAll();

    $entity_ids = [];
    foreach ($results as $result) {
      $entity_ids[] = $result->entity_id;
    }

    if (!empty($entity_ids)) {
      // Load entities in batch for better performance.
      $loaded_entities = $this->entityTypeManager
        ->getStorage($entity_type)
        ->loadMultiple($entity_ids);

      foreach ($loaded_entities as $entity) {
        $entities[] = [
          'entity' => $entity,
          'revision_type' => 'current',
          'is_published' => $this->getEntityPublishedStatus($entity),
          'revision_id' => method_exists($entity, 'getRevisionId') ? $entity->getRevisionId() : NULL,
        ];
      }
    }

    // Check drafts if enabled and we haven't hit the limit.
    if ($check_drafts && count($entities) < $limit) {
      if ($entity_type === 'node') {
        // For nodes, check draft revisions.
        $draft_entities = $this->findDraftEntitiesOptimized($field_name, $tid, $limit - count($entities));
        $entities = array_merge($entities, $draft_entities);
      }
      elseif ($entity_type === 'paragraph') {
        // For paragraphs, find those whose parent entities are in draft state.
        if ($entity_type === 'paragraph' && $this->isParagraphsModuleAvailable()) {
          $draft_paragraphs = $this->findDraftParagraphsOptimized($field_name, $tid, $limit - count($entities));
          $entities = array_merge($entities, $draft_paragraphs);
        }
      }
    }

    return array_slice($entities, 0, $limit);
  }

  /**
   * Find draft revisions using optimized query.
   *
   * @param string $field_name
   *   The field name.
   * @param int $tid
   *   The taxonomy term ID.
   * @param int $limit
   *   Maximum number of results.
   *
   * @return array
   *   Array of draft entity usage data.
   */
  protected function findDraftEntitiesOptimized($field_name, $tid, $limit) {
    $entities = [];
    $revision_table = "node_revision__{$field_name}";

    if (!$this->database->schema()->tableExists($revision_table)) {
      return [];
    }

    // Join with node_field_revision to check published status.
    $query = $this->database->select($revision_table, 'r');
    $query->join('node_field_revision', 'nfr', 'r.revision_id = nfr.vid');
    $query->fields('r', ['entity_id', 'revision_id']);
    $query->condition("r.{$field_name}_target_id", $tid);
    // Unpublished only.
    $query->condition('nfr.status', 0);
    $query->range(0, $limit);

    $results = $query->execute()->fetchAll();

    foreach ($results as $result) {
      try {
        $revision = $this->entityTypeManager
          ->getStorage('node')
          ->loadRevision($result->revision_id);

        if ($revision) {
          $entities[] = [
            'entity' => $revision,
            'revision_type' => 'draft',
            'is_published' => FALSE,
            'revision_id' => $revision->getRevisionId(),
          ];
        }
      }
      catch (\Exception $e) {
        // Skip if revision can't be loaded.
        continue;
      }
    }

    return $entities;
  }

  /**
   * Get total usage count for a taxonomy term (optimized).
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return int
   *   Total number of entities using this term.
   */
  public function getUsageCount($tid) {
    $term = Term::load($tid);
    if (!$term) {
      return 0;
    }

    $config = $this->configFactory->get('taxonomy_usage.settings');
    $enabled_entity_types = $config->get('entity_types') ?? ['node'];

    $count = 0;
    $taxonomy_fields = $this->getTaxonomyFieldsForVocabulary($term->bundle(), $enabled_entity_types);

    foreach ($taxonomy_fields as $field_info) {
      $field_name = $field_info['field_name'];
      $entity_type = $field_info['entity_type'];

      // Count using database query for performance.
      $data_table = "{$entity_type}__{$field_name}";
      if ($this->database->schema()->tableExists($data_table)) {
        $query = $this->database->select($data_table, 'f');
        $query->condition("f.{$field_name}_target_id", $tid);
        $count += $query->countQuery()->execute()->fetchField();
      }
    }

    return $count;
  }

  /**
   * Check if a taxonomy term is in use (optimized).
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return bool
   *   TRUE if term is in use, FALSE otherwise.
   */
  public function isTermInUse($tid) {
    $term = Term::load($tid);
    if (!$term) {
      return FALSE;
    }

    $config = $this->configFactory->get('taxonomy_usage.settings');
    $enabled_entity_types = $config->get('entity_types') ?? ['node'];

    $taxonomy_fields = $this->getTaxonomyFieldsForVocabulary($term->bundle(), $enabled_entity_types);

    foreach ($taxonomy_fields as $field_info) {
      $field_name = $field_info['field_name'];
      $entity_type = $field_info['entity_type'];

      // Check using database query for performance.
      $data_table = $this->getSafeTableName($entity_type, $field_name);
      if ($data_table && $this->database->schema()->tableExists($data_table)) {
        $query = $this->database->select($data_table, 'f');
        $query->fields('f', ['entity_id']);
        $query->condition("f.{$field_name}_target_id", $tid);
        $query->range(0, 1);

        if ($query->execute()->fetchField()) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

  /**
   * Get comprehensive taxonomy term usage including child terms.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return array
   *   Structured array with 'direct' and 'children' usage data.
   */
  public function getTermUsageWithChildren($tid) {
    $term = Term::load($tid);
    if (!$term) {
      return ['direct' => [], 'children' => []];
    }

    // Get direct usage.
    $direct_usage = $this->getTermUsageByField($tid);

    // Get child term usage.
    $children_usage = [];
    $child_terms = $this->entityTypeManager
      ->getStorage('taxonomy_term')
      ->loadTree($term->bundle(), $tid, NULL, TRUE);

    if (!empty($child_terms)) {
      foreach ($child_terms as $child_term) {
        $child_usage = $this->getTermUsageByField($child_term->id());
        if (!empty($child_usage)) {
          $children_usage[$child_term->id()] = [
            'term' => $child_term,
            'usage' => $child_usage,
          ];
        }
      }
    }

    return [
      'direct' => $direct_usage,
      'children' => $children_usage,
    ];
  }

  /**
   * Get usage summary by entity type for a taxonomy term.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return array
   *   Array of usage counts keyed by entity type.
   */
  public function getUsageSummaryByEntityType($tid) {
    $term = Term::load($tid);
    if (!$term) {
      return [];
    }

    $config = $this->configFactory->get('taxonomy_usage.settings');
    $enabled_entity_types = $config->get('entity_types') ?? ['node'];

    $summary = [];
    $taxonomy_fields = $this->getTaxonomyFieldsForVocabulary($term->bundle(), $enabled_entity_types);

    foreach ($taxonomy_fields as $field_info) {
      $field_name = $field_info['field_name'];
      $entity_type = $field_info['entity_type'];

      // Count using database query for performance.
      $data_table = $this->getSafeTableName($entity_type, $field_name);
      if ($data_table && $this->database->schema()->tableExists($data_table)) {
        $query = $this->database->select($data_table, 'f');
        $query->condition("f.{$field_name}_target_id", $tid);
        $count = $query->countQuery()->execute()->fetchField();

        if ($count > 0) {
          if (!isset($summary[$entity_type])) {
            $summary[$entity_type] = 0;
          }
          $summary[$entity_type] += $count;
        }
      }
    }

    return $summary;
  }

  /**
   * Clear cache for a specific term.
   *
   * @param int $tid
   *   The taxonomy term ID.
   */
  public function clearTermCache($tid) {
    $this->cache->delete("taxonomy_usage:term:{$tid}");
  }

  /**
   * Get the proper published status for an entity.
   *
   * For paragraphs, this recursively checks the parent entity's status.
   * For other entities, it checks their own published status.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   TRUE if published, FALSE if not.
   */
  protected function getEntityPublishedStatus($entity) {
    // For paragraphs, check the parent entity's status.
    if ($entity->getEntityTypeId() === 'paragraph' && $this->isParagraphsModuleAvailable()) {
      return $this->getParagraphPublishedStatus($entity);
    }

    // For other entities, use standard method.
    if (method_exists($entity, 'isPublished')) {
      return $entity->isPublished();
    }

    // Default to TRUE if no published status method.
    return TRUE;
  }

  /**
   * Get published status for paragraph by checking parent entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $paragraph
   *   The paragraph entity.
   *
   * @return bool
   *   TRUE if parent is published, FALSE otherwise.
   */
  protected function getParagraphPublishedStatus($paragraph) {
    try {
      // Method 1: Try getParentEntity() if available.
      if (method_exists($paragraph, 'getParentEntity')) {
        $parent = $paragraph->getParentEntity();
        if ($parent) {
          return $this->getEntityPublishedStatus($parent);
        }
      }

      // Method 2: Try parent_id/parent_type fields.
      if (method_exists($paragraph, 'hasField') &&
          $paragraph->hasField('parent_id') &&
          $paragraph->hasField('parent_type')) {

        $parent_id = $paragraph->get('parent_id')->value;
        $parent_type = $paragraph->get('parent_type')->value;

        if ($parent_id && $parent_type) {
          $parent_storage = $this->entityTypeManager->getStorage($parent_type);
          $parent = $parent_storage->load($parent_id);

          if ($parent) {
            return $this->getEntityPublishedStatus($parent);
          }
        }
      }

      // Method 3: Database query fallback.
      if ($this->tableExists('paragraphs_item_field_data')) {
        $query = $this->database->select('paragraphs_item_field_data', 'p')
          ->fields('p', ['parent_id', 'parent_type'])
          ->condition('p.id', $paragraph->id())
          ->range(0, 1);

        $result = $query->execute()->fetchAssoc();
      }

      if ($result && $result['parent_id'] && $result['parent_type']) {
        $parent_storage = $this->entityTypeManager->getStorage($result['parent_type']);
        $parent = $parent_storage->load($result['parent_id']);

        if ($parent) {
          return $this->getEntityPublishedStatus($parent);
        }
      }

    }
    catch (\Exception $e) {
      // If all methods fail, default to FALSE for safety.
    }

    // Default to FALSE if we can't determine parent status.
    return FALSE;
  }

  /**
   * Find paragraphs whose parent entities are in draft state.
   *
   * @param string $field_name
   *   The field name.
   * @param int $tid
   *   The taxonomy term ID.
   * @param int $limit
   *   Maximum number of results.
   *
   * @return array
   *   Array of paragraph entities with draft parent entities.
   */
  protected function findDraftParagraphsOptimized($field_name, $tid, $limit) {
    // Return empty array if paragraphs module is not available.
    if (!$this->isParagraphsModuleAvailable()) {
      return [];
    }

    $entities = [];
    $data_table = $this->getSafeTableName('paragraph', $field_name);
    $revision_table = $this->getSafeTableName('paragraph', $field_name, TRUE);

    // First, check current paragraph data table for paragraphs with draft parents.
    if ($data_table && $this->tableExists($data_table)) {
      $entities = array_merge($entities, $this->findCurrentParagraphsWithDraftParents($data_table, $field_name, $tid, $limit));
    }

    // Then, check revision table for paragraphs that only exist in draft revisions.
    if ($revision_table && count($entities) < $limit && $this->tableExists($revision_table)) {
      $remaining_limit = $limit - count($entities);
      $revision_entities = $this->findDraftRevisionParagraphs($revision_table, $field_name, $tid, $remaining_limit);
      $entities = array_merge($entities, $revision_entities);
    }

    return array_slice($entities, 0, $limit);
  }

  /**
   * Find current paragraphs whose parent entities are in draft state.
   */
  protected function findCurrentParagraphsWithDraftParents($data_table, $field_name, $tid, $limit) {
    $entities = [];

    // Find paragraphs using the term.
    $query = $this->database->select($data_table, 'p');
    $query->fields('p', ['entity_id']);
    $query->condition("p.{$field_name}_target_id", $tid);
    // Get more to account for filtering.
    $query->range(0, $limit * 3);

    $paragraph_ids = $query->execute()->fetchCol();

    if (empty($paragraph_ids)) {
      return [];
    }

    // Load paragraphs and check their parent status.
    $paragraphs = $this->entityTypeManager
      ->getStorage('paragraph')
      ->loadMultiple($paragraph_ids);

    $found_count = 0;
    foreach ($paragraphs as $paragraph) {
      if ($found_count >= $limit) {
        break;
      }

      // Check if this paragraph's ultimate parent is in draft state.
      $ultimate_parent = $this->findUltimateParentEntity($paragraph);

      if ($ultimate_parent) {
        // Check if parent is unpublished/draft.
        $is_parent_published = $this->getEntityPublishedStatus($ultimate_parent);

        if (!$is_parent_published) {
          $entities[] = [
            'entity' => $paragraph,
            'revision_type' => 'draft',
            'is_published' => FALSE,
            'revision_id' => method_exists($paragraph, 'getRevisionId') ? $paragraph->getRevisionId() : NULL,
          ];
          $found_count++;
        }
      }
    }

    return $entities;
  }

  /**
   * Find paragraphs that only exist in draft revisions.
   */
  protected function findDraftRevisionParagraphs($revision_table, $field_name, $tid, $limit) {
    $entities = [];

    // Find paragraph revisions using the term.
    $query = $this->database->select($revision_table, 'pr');
    $query->fields('pr', ['entity_id', 'revision_id']);
    $query->condition("pr.{$field_name}_target_id", $tid);
    // Get more to account for filtering.
    $query->range(0, $limit * 10);

    $results = $query->execute()->fetchAll();

    if (empty($results)) {
      return [];
    }

    $found_count = 0;
    $processed_paragraphs = [];

    foreach ($results as $result) {
      if ($found_count >= $limit) {
        break;
      }

      // Skip if we already processed this paragraph.
      if (isset($processed_paragraphs[$result->entity_id])) {
        continue;
      }

      try {
        // Load the paragraph revision.
        $paragraph_revision = $this->entityTypeManager
          ->getStorage('paragraph')
          ->loadRevision($result->revision_id);

        if (!$paragraph_revision) {
          continue;
        }

        // Find which node revision contains this paragraph revision.
        $node_revision_info = $this->findNodeRevisionForParagraph($paragraph_revision);

        if ($node_revision_info && !$node_revision_info['is_published']) {
          $entities[] = [
            'entity' => $paragraph_revision,
            'revision_type' => 'draft',
            'is_published' => FALSE,
            'revision_id' => $result->revision_id,
          ];
          $found_count++;
          $processed_paragraphs[$result->entity_id] = TRUE;
        }
      }
      catch (\Exception $e) {
        // Skip if revision can't be loaded.
        continue;
      }
    }

    return $entities;
  }

  /**
   * Find which node revision contains a specific paragraph revision.
   *
   * @param \Drupal\Core\Entity\EntityInterface $paragraph_revision
   *   The paragraph revision.
   *
   * @return array|null
   *   Array with node revision info or null.
   */
  protected function findNodeRevisionForParagraph($paragraph_revision) {
    try {
      // Traverse up the hierarchy to find the ultimate parent node.
      $current_entity = $paragraph_revision;
      $depth = 0;

      while ($depth < 10 && $current_entity->getEntityTypeId() === 'paragraph') {
        $parent_id = $current_entity->get('parent_id')->value;
        $parent_type = $current_entity->get('parent_type')->value;

        if (!$parent_id || !$parent_type) {
          break;
        }

        if ($parent_type === 'node') {
          // We found the parent node. Now we need to determine which node revision
          // this paragraph revision belongs to. This is tricky because we need to
          // find the node revision that was active when this paragraph revision was created.
          // Load the node and check its revisions.
          $node = $this->entityTypeManager->getStorage('node')->load($parent_id);
          if (!$node) {
            break;
          }

          // Get all node revisions and find the one that likely contains this paragraph revision.
          $node_revisions = $this->database->select('node_field_revision', 'nfr')
            ->fields('nfr', ['vid', 'status', 'changed'])
            ->condition('nfr.nid', $parent_id)
            ->orderBy('nfr.vid', 'DESC')
            ->execute()
            ->fetchAll();

          // For now, let's check if there are any unpublished revisions.
          foreach ($node_revisions as $revision) {
            if ($revision->status == 0) {
              return [
                'node_id' => $parent_id,
                'node_revision_id' => $revision->vid,
                'is_published' => FALSE,
              ];
            }
          }

          // If no unpublished revisions found, this paragraph is in published content.
          return [
            'node_id' => $parent_id,
            'node_revision_id' => $node->getRevisionId(),
            'is_published' => TRUE,
          ];
        }
        elseif ($parent_type === 'paragraph') {
          // Load the parent paragraph revision.
          // We need to find the revision of the parent paragraph that was active
          // when this paragraph revision was created.
          $parent_paragraph = $this->entityTypeManager->getStorage('paragraph')->load($parent_id);
          if ($parent_paragraph) {
            $current_entity = $parent_paragraph;
            $depth++;
          }
          else {
            break;
          }
        }
        else {
          // Other entity types (media, etc.) - for now, assume published.
          return [
            'node_id' => $parent_id,
            'node_revision_id' => NULL,
            'is_published' => TRUE,
          ];
        }
      }
    }
    catch (\Exception $e) {
      // If anything fails, assume published.
    }

    return NULL;
  }

  /**
   * Find the ultimate parent entity for a paragraph.
   *
   * @param \Drupal\Core\Entity\EntityInterface $paragraph
   *   The paragraph entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The ultimate parent entity or null.
   */
  protected function findUltimateParentEntity($paragraph) {
    $depth = 0;
    $current_entity = $paragraph;

    while ($depth < 10 && $current_entity->getEntityTypeId() === 'paragraph') {
      try {
        // Try getParentEntity() method.
        if (method_exists($current_entity, 'getParentEntity')) {
          $parent = $current_entity->getParentEntity();
          if ($parent) {
            $current_entity = $parent;
            $depth++;
            continue;
          }
        }

        // Try database lookup.
        $query = $this->database->select('paragraphs_item_field_data', 'p')
          ->fields('p', ['parent_id', 'parent_type'])
          ->condition('p.id', $current_entity->id())
          ->range(0, 1);

        $result = $query->execute()->fetchAssoc();

        if ($result && $result['parent_id'] && $result['parent_type']) {
          $parent_storage = $this->entityTypeManager->getStorage($result['parent_type']);
          $parent = $parent_storage->load($result['parent_id']);

          if ($parent) {
            $current_entity = $parent;
            $depth++;
            continue;
          }
        }

        // No parent found, break the loop.
        break;
      }
      catch (\Exception $e) {
        break;
      }
    }

    // Return the ultimate parent if it's not a paragraph.
    if ($current_entity->getEntityTypeId() !== 'paragraph') {
      return $current_entity;
    }

    return NULL;
  }

}
