<?php

namespace Drupal\taxonomy_usage;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Service for displaying taxonomy term usage information.
 */
class TaxonomyUsageDisplay {

  use StringTranslationTrait;

  /**
   * The taxonomy usage detector service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDetector
   */
  protected $usageDetector;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Constructs a TaxonomyUsageDisplay object.
   *
   * @param \Drupal\taxonomy_usage\TaxonomyUsageDetector $usage_detector
   *   The taxonomy usage detector service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   */
  public function __construct(TaxonomyUsageDetector $usage_detector, TranslationInterface $string_translation, ModuleHandlerInterface $module_handler, EntityTypeManagerInterface $entity_type_manager, Connection $database) {
    $this->usageDetector = $usage_detector;
    $this->stringTranslation = $string_translation;
    $this->moduleHandler = $module_handler;
    $this->entityTypeManager = $entity_type_manager;
    $this->database = $database;
  }

  /**
   * Check if the paragraphs module is available.
   *
   * @return bool
   *   TRUE if paragraphs module is enabled, FALSE otherwise.
   */
  protected function isParagraphsModuleAvailable() {
    return $this->moduleHandler->moduleExists('paragraphs');
  }

  /**
   * Render usage data as HTML table.
   *
   * @param int $tid
   *   The taxonomy term ID.
   * @param bool $show_term_name
   *   Whether to show the term name in field headers.
   * @param bool $include_children
   *   Whether to include child term usage.
   *
   * @return string
   *   HTML markup for the usage display.
   */
  public function renderUsageTable($tid, $show_term_name = FALSE, $include_children = TRUE) {
    if ($include_children) {
      $all_usage_data = $this->usageDetector->getTermUsageWithChildren($tid);
      $usage_data = $all_usage_data['direct'];
      $children_data = $all_usage_data['children'];
    }
    else {
      $usage_data = $this->usageDetector->getTermUsageByField($tid);
      $children_data = [];
    }

    if (empty($usage_data) && empty($children_data)) {
      return '<p>' . $this->t('No content found using this taxonomy term or its children.') . '</p>';
    }

    $output = '<div class="taxonomy-usage-display">';

    // Add entity type summary for better overview.
    $summary = $this->usageDetector->getUsageSummaryByEntityType($tid);
    if (!empty($summary)) {
      $output .= '<div class="taxonomy-usage-summary">';
      $output .= '<p><strong>' . $this->t('Usage Summary:') . '</strong> ';
      $summary_parts = [];
      foreach ($summary as $entity_type => $count) {
        $entity_type_label = ucfirst(str_replace('_', ' ', $entity_type));

        // Handle pluralization properly for different entity types.
        $plural_label = $entity_type_label;
        if ($count != 1) {
          // Words that are already plural or have special plural forms.
          $special_plurals = [
            'media' => 'media',
          ];

          $lowercase_label = strtolower($entity_type_label);
          if (isset($special_plurals[$lowercase_label])) {
            // Keep as is.
            $plural_label = $entity_type_label;
          }
          else {
            // Add 's' for regular words.
            $plural_label = $entity_type_label . 's';
          }
        }

        $summary_parts[] = $this->t('@count @type', [
          '@count' => $count,
          '@type' => $plural_label,
        ]);
      }
      $output .= implode(', ', $summary_parts) . '</p>';
      $output .= '</div>';
    }

    // Create tabbed interface if we have children usage, or both direct and children.
    if (!empty($children_data)) {
      $output .= $this->renderTabbedUsage($usage_data, $children_data, $show_term_name, $tid);
    }
    else {
      // Show warning and direct usage only (no children).
      $output .= '<p class="taxonomy-usage-warning"><strong>' . $this->t('Warning:') . '</strong> ' . $this->t('The following content is using this taxonomy term. Please remove all references before deleting the term.') . '</p>';
      $output .= $this->renderDirectUsage($usage_data, $show_term_name, $tid);
    }

    $output .= '</div>';

    return $output;
  }

  /**
   * Render tabbed interface for usage data.
   */
  protected function renderTabbedUsage($usage_data, $children_data, $show_term_name, $tid) {
    $output = '<div class="taxonomy-usage-tabs">';

    // Tab navigation.
    $output .= '<ul class="taxonomy-usage-tab-nav">';

    // Only show direct usage tab if there is direct usage.
    if (!empty($usage_data)) {
      $output .= '<li class="taxonomy-usage-tab-item active">';
      $output .= '<a href="#direct-usage" class="taxonomy-usage-tab-link">' . $this->t('Direct Usage') . '</a>';
      $output .= '</li>';
    }

    // Show child usage tab (make it active if no direct usage).
    $child_tab_class = empty($usage_data) ? 'taxonomy-usage-tab-item active' : 'taxonomy-usage-tab-item';
    $output .= '<li class="' . $child_tab_class . '">';
    $output .= '<a href="#child-usage" class="taxonomy-usage-tab-link">' . $this->t('Child Term Usage') . '</a>';
    $output .= '</li>';
    $output .= '</ul>';

    // Tab content.
    $output .= '<div class="taxonomy-usage-tab-content">';

    // Direct usage tab (only if there is direct usage).
    if (!empty($usage_data)) {
      $output .= '<div id="direct-usage" class="taxonomy-usage-tab-pane active">';
      $output .= '<p class="taxonomy-usage-warning"><strong>' . $this->t('Warning:') . '</strong> ' . $this->t('The following content is using this taxonomy term. Please remove all references before deleting the term.') . '</p>';
      $output .= $this->renderDirectUsage($usage_data, $show_term_name, $tid);
      $output .= '</div>';
    }

    // Child usage tab (make it active if no direct usage).
    $child_pane_class = empty($usage_data) ? 'taxonomy-usage-tab-pane active' : 'taxonomy-usage-tab-pane';
    $output .= '<div id="child-usage" class="' . $child_pane_class . '">';
    $output .= '<p class="taxonomy-usage-warning"><strong>' . $this->t('Warning:') . '</strong> ' . $this->t('The following content is using child terms of this taxonomy term. Please remove all references before deleting the parent term.') . '</p>';
    $output .= $this->renderChildUsage($children_data);
    $output .= '</div>';

    // tab-content.
    $output .= '</div>';
    // Tabs.
    $output .= '</div>';

    return $output;
  }

  /**
   * Render direct usage data.
   */
  protected function renderDirectUsage($usage_data, $show_term_name, $tid) {
    $output = '';

    foreach ($usage_data as $field_data) {
      $output .= '<div class="taxonomy-usage-field-group">';

      if ($show_term_name) {
        // Load term to get name for header.
        $term = $this->entityTypeManager->getStorage('taxonomy_term')->load($tid);
        $term_name = $term ? $term->label() : $this->t('Unknown term');
        $output .= '<h3>' . $this->t('Content using "@term_name" in @field field (@bundle):', [
          '@term_name' => $term_name,
          '@field' => $field_data['field_label'],
          '@bundle' => $field_data['bundle_label'],
        ]) . '</h3>';
      }
      else {
        $output .= '<h3>' . $this->t('Content using term in @field field (@bundle):', [
          '@field' => $field_data['field_label'],
          '@bundle' => $field_data['bundle_label'],
        ]) . '</h3>';
      }

      $output .= '<table class="taxonomy-usage-table">';
      $output .= '<thead><tr>';
      $output .= '<th>' . $this->t('Title') . '</th>';
      $output .= '<th>' . $this->t('Type') . '</th>';
      $output .= '<th>' . $this->t('Status') . '</th>';
      $output .= '<th>' . $this->t('Revision') . '</th>';
      $output .= '<th>' . $this->t('Actions') . '</th>';
      $output .= '</tr></thead><tbody>';

      foreach ($field_data['entities'] as $entity_data) {
        $entity = $entity_data['entity'];
        $output .= '<tr>';

        // Handle different entity types that may not have canonical URLs.
        $entity_link = $this->getEntityLink($entity);
        $output .= '<td>' . $entity_link . '</td>';

        $output .= '<td>' . $entity->bundle() . '</td>';
        $output .= '<td>' . $this->getEntityStatusText($entity, $entity_data['is_published']) . '</td>';
        $output .= '<td>' . $this->t('@type (ID: @id)', [
          '@type' => ucfirst($entity_data['revision_type']),
          '@id' => $entity_data['revision_id'] ?? 'N/A',
        ]) . '</td>';
        $output .= '<td>';

        // Handle edit links for different entity types.
        $edit_link = $this->getEntityEditLink($entity);
        $output .= $edit_link;

        $output .= '</td>';
        $output .= '</tr>';
      }

      $output .= '</tbody></table>';
      $output .= '</div>';
    }

    return $output;
  }

  /**
   * Render child usage data.
   */
  protected function renderChildUsage($children_data) {
    $output = '';

    foreach ($children_data as $child_info) {
      $child_term = $child_info['term'];
      $child_usage = $child_info['usage'];

      $output .= '<div class="taxonomy-usage-child-term">';
      $output .= '<h3 class="taxonomy-usage-child-term-name">' . $this->t('Child term: "@child_name"', ['@child_name' => $child_term->label()]) . '</h3>';

      foreach ($child_usage as $field_data) {
        $output .= '<div class="taxonomy-usage-field-group">';
        $output .= '<h3>' . $this->t('Content using "@child_name" in @field field (@bundle):', [
          '@child_name' => $child_term->label(),
          '@field' => $field_data['field_label'],
          '@bundle' => $field_data['bundle_label'],
        ]) . '</h3>';

        $output .= '<table class="taxonomy-usage-table">';
        $output .= '<thead><tr>';
        $output .= '<th>' . $this->t('Title') . '</th>';
        $output .= '<th>' . $this->t('Type') . '</th>';
        $output .= '<th>' . $this->t('Status') . '</th>';
        $output .= '<th>' . $this->t('Revision') . '</th>';
        $output .= '<th>' . $this->t('Actions') . '</th>';
        $output .= '</tr></thead><tbody>';

        foreach ($field_data['entities'] as $entity_data) {
          $entity = $entity_data['entity'];
          $output .= '<tr>';

          // Handle different entity types that may not have canonical URLs.
          $entity_link = $this->getEntityLink($entity);
          $output .= '<td>' . $entity_link . '</td>';

          $output .= '<td>' . $entity->bundle() . '</td>';
          $output .= '<td>' . $this->getEntityStatusText($entity, $entity_data['is_published']) . '</td>';
          $output .= '<td>' . $this->t('@type (ID: @id)', [
            '@type' => ucfirst($entity_data['revision_type']),
            '@id' => $entity_data['revision_id'] ?? 'N/A',
          ]) . '</td>';
          $output .= '<td>';

          // Handle edit links for different entity types.
          $edit_link = $this->getEntityEditLink($entity);
          $output .= $edit_link;

          $output .= '</td>';
          $output .= '</tr>';
        }

        $output .= '</tbody></table>';
        $output .= '</div>';
      }

      $output .= '</div>';
    }

    return $output;
  }

  /**
   * Render usage data as a simple list for emails or text display.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return string
   *   Plain text list of usage.
   */
  public function renderUsageList($tid) {
    $usage_data = $this->usageDetector->getTermUsageByField($tid);

    if (empty($usage_data)) {
      return $this->t('No content found using this taxonomy term.');
    }

    $output = $this->t('Content using this taxonomy term:');
    $output .= "\n\n";

    foreach ($usage_data as $field_data) {
      $output .= $this->t('Field: @field (@bundle)', [
        '@field' => $field_data['field_label'],
        '@bundle' => $field_data['bundle_label'],
      ]);
      $output .= "\n";

      foreach ($field_data['entities'] as $entity_data) {
        $entity = $entity_data['entity'];
        $status = $this->getEntityStatusText($entity, $entity_data['is_published']);
        $output .= "  - " . $entity->label() . " (" . $entity->bundle() . ", " . $status . ")\n";
      }

      $output .= "\n";
    }

    return $output;
  }

  /**
   * Get usage count for validation.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return int
   *   Total number of entities using this term.
   */
  public function getUsageCount($tid) {
    return $this->usageDetector->getUsageCount($tid);
  }

  /**
   * Check if a taxonomy term is in use.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return bool
   *   TRUE if term is in use, FALSE otherwise.
   */
  public function isTermInUse($tid) {
    return $this->usageDetector->isTermInUse($tid);
  }

  /**
   * Get a safe link for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string
   *   HTML link or plain text if no URL available.
   */
  protected function getEntityLink($entity) {
    try {
      // For paragraphs, always try to link to the parent entity first.
      if ($entity->getEntityTypeId() === 'paragraph' && $this->isParagraphsModuleAvailable()) {
        try {
          $parent = $entity->getParentEntity();
          if ($parent && $parent->hasLinkTemplate('canonical')) {
            // Create a more descriptive link showing paragraph type and parent.
            $paragraph_label = $entity->label() ?: $this->t('Paragraph @id', ['@id' => $entity->id()]);
            $paragraph_type = $entity->bundle();
            return '<a href="' . $parent->toUrl()->toString() . '">' . $paragraph_label . '</a> <small>(' . $paragraph_type . ' in ' . $parent->label() . ')</small>';
          }
        }
        catch (\Exception $e) {
          // Parent method not available or failed, continue to fallback.
        }

        // Fallback for paragraphs without accessible parent.
        $paragraph_label = $entity->label() ?: $this->t('Paragraph @id', ['@id' => $entity->id()]);
        return $paragraph_label . ' <small>(' . $entity->bundle() . ')</small>';
      }

      // For other entity types, try to get canonical URL.
      if ($entity->hasLinkTemplate('canonical')) {
        return '<a href="' . $entity->toUrl()->toString() . '">' . $entity->label() . '</a>';
      }

      // Fallback to just the label for other entity types.
      return $entity->label() . ' (' . $this->t('ID: @id', ['@id' => $entity->id()]) . ')';
    }
    catch (\Exception $e) {
      // If anything fails, just return the label.
      return $entity->label() . ' (' . $this->t('ID: @id', ['@id' => $entity->id()]) . ')';
    }
  }

  /**
   * Get a safe edit link for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   *
   * @return string
   *   HTML edit link or empty string if not available.
   */
  protected function getEntityEditLink($entity) {
    // For paragraphs, try to get parent entity link.
    if ($entity->getEntityTypeId() === 'paragraph' && $this->isParagraphsModuleAvailable()) {
      return $this->getParagraphParentLink($entity);
    }

    // For other entity types, standard approach.
    try {
      if (!$entity->access('update')) {
        return '';
      }

      if ($entity->hasLinkTemplate('edit-form')) {
        return '<a href="' . $entity->toUrl('edit-form')->toString() . '">' . $this->t('Edit') . '</a>';
      }

      return '';
    }
    catch (\Exception $e) {
      return '';
    }
  }

  /**
   * Get parent entity link for paragraphs.
   *
   * @param \Drupal\Core\Entity\EntityInterface $paragraph
   *   The paragraph entity.
   *
   * @return string
   *   HTML link or fallback text.
   */
  protected function getParagraphParentLink($paragraph) {
    try {
      // Find the ultimate parent (traverse up the hierarchy).
      $ultimate_parent = $this->findUltimateParent($paragraph);

      if ($ultimate_parent) {
        return $this->createParentLink($ultimate_parent);
      }
    }
    catch (\Exception $e) {
      // All methods failed.
    }

    // Final fallback.
    return '<em>' . $this->t('Edit via parent') . '</em>';
  }

  /**
   * Recursively find the ultimate parent entity (node, media, etc.).
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to find parent for.
   * @param int $depth
   *   Current recursion depth (prevent infinite loops).
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The ultimate parent entity or null.
   */
  protected function findUltimateParent($entity, $depth = 0) {
    // Prevent infinite recursion.
    if ($depth > 10) {
      return NULL;
    }

    // If this isn't a paragraph, it's likely our ultimate parent.
    if ($entity->getEntityTypeId() !== 'paragraph') {
      return $entity;
    }

    // Try multiple methods to find parent.
    $parent = NULL;

    // Method 1: Try getParentEntity() if available.
    if (method_exists($entity, 'getParentEntity')) {
      try {
        $parent = $entity->getParentEntity();
      }
      catch (\Exception $e) {
        // Continue to next method.
      }
    }

    // Method 2: Try parent_id/parent_type fields.
    if (!$parent && method_exists($entity, 'hasField')) {
      try {
        if ($entity->hasField('parent_id') && $entity->hasField('parent_type')) {
          $parent_id = $entity->get('parent_id')->value;
          $parent_type = $entity->get('parent_type')->value;

          if ($parent_id && $parent_type) {
            $parent_storage = $this->entityTypeManager->getStorage($parent_type);
            $parent = $parent_storage->load($parent_id);
          }
        }
      }
      catch (\Exception $e) {
        // Continue to next method.
      }
    }

    // Method 3: Database query fallback.
    if (!$parent && $this->isParagraphsModuleAvailable()) {
      try {
        $database = $this->database;
        if ($database->schema()->tableExists('paragraphs_item_field_data')) {
          $query = $database->select('paragraphs_item_field_data', 'p')
            ->fields('p', ['parent_id', 'parent_type'])
            ->condition('p.id', $entity->id())
            ->range(0, 1);

          $result = $query->execute()->fetchAssoc();

          if ($result && $result['parent_id'] && $result['parent_type']) {
            $parent_storage = $this->entityTypeManager->getStorage($result['parent_type']);
            $parent = $parent_storage->load($result['parent_id']);
          }
        }
      }
      catch (\Exception $e) {
        // Database query failed.
      }
    }

    // If we found a parent, recursively check if it has a parent too.
    if ($parent) {
      return $this->findUltimateParent($parent, $depth + 1);
    }

    // No parent found.
    return NULL;
  }

  /**
   * Create a link to parent entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $parent
   *   The parent entity.
   *
   * @return string
   *   HTML link.
   */
  protected function createParentLink($parent) {
    try {
      $parent_type = $parent->getEntityTypeId();
      $parent_label = ucfirst($parent_type);

      // Try edit link first.
      if ($parent->hasLinkTemplate('edit-form')) {
        $edit_url = $parent->toUrl('edit-form')->toString();
        return '<a href="' . $edit_url . '">' . $this->t('Edit @parent_type', ['@parent_type' => $parent_label]) . '</a>';
      }

      // Try view link.
      if ($parent->hasLinkTemplate('canonical')) {
        $view_url = $parent->toUrl()->toString();
        return '<a href="' . $view_url . '">' . $this->t('View @parent_type', ['@parent_type' => $parent_label]) . '</a>';
      }

      // Show parent info without link.
      return $this->t('Parent: @parent_label', ['@parent_label' => $parent->label()]);
    }
    catch (\Exception $e) {
      return '<em>' . $this->t('Edit via parent') . '</em>';
    }
  }

  /**
   * Get descriptive status text for an entity.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity.
   * @param bool $is_published
   *   Whether the entity (or its parent) is published.
   *
   * @return string
   *   Descriptive status text.
   */
  protected function getEntityStatusText($entity, $is_published) {
    // For paragraphs, make it clear we're showing parent status.
    if ($entity->getEntityTypeId() === 'paragraph') {
      $parent_type = $this->getParentEntityType($entity);
      $parent_label = ucfirst($parent_type ?: 'parent');

      if ($is_published) {
        return $this->t('@parent_type Published', ['@parent_type' => $parent_label]);
      }
      else {
        return $this->t('@parent_type Draft', ['@parent_type' => $parent_label]);
      }
    }

    // For other entities, use standard text.
    if ($is_published) {
      return $this->t('Published');
    }
    else {
      return $this->t('Draft');
    }
  }

  /**
   * Get the parent entity type for a paragraph.
   *
   * @param \Drupal\Core\Entity\EntityInterface $paragraph
   *   The paragraph entity.
   *
   * @return string
   *   The parent entity type or 'parent' as fallback.
   */
  protected function getParentEntityType($paragraph) {
    try {
      // Find the ultimate parent to get its type.
      $ultimate_parent = $this->findUltimateParent($paragraph);

      if ($ultimate_parent) {
        return $ultimate_parent->getEntityTypeId();
      }
    }
    catch (\Exception $e) {
      // If we can't find the parent, try alternative methods.
    }

    // Fallback: try to get parent type from database.
    try {
      $query = $this->database->select('paragraphs_item_field_data', 'p')
        ->fields('p', ['parent_type'])
        ->condition('p.id', $paragraph->id())
        ->range(0, 1);

      $result = $query->execute()->fetchAssoc();

      if ($result && $result['parent_type']) {
        return $result['parent_type'];
      }
    }
    catch (\Exception $e) {
      // Database query failed.
    }

    // Final fallback.
    return 'parent';
  }

}
