<?php

namespace Drupal\taxonomy_usage;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for validating taxonomy term deletion.
 */
class TaxonomyUsageValidator {

  use StringTranslationTrait;

  /**
   * The taxonomy usage detector service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDetector
   */
  protected $usageDetector;

  /**
   * Constructs a TaxonomyUsageValidator object.
   *
   * @param \Drupal\taxonomy_usage\TaxonomyUsageDetector $usage_detector
   *   The taxonomy usage detector service.
   */
  public function __construct(TaxonomyUsageDetector $usage_detector) {
    $this->usageDetector = $usage_detector;
  }

  /**
   * Validate if a taxonomy term can be deleted.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return array
   *   Array with 'valid' boolean and 'message' string.
   */
  public function validateTermDeletion($tid) {
    $usage_count = $this->usageDetector->getUsageCount($tid);

    if ($usage_count > 0) {
      return [
        'valid' => FALSE,
        'message' => $this->t('Cannot delete taxonomy term because it is referenced by @count content items. Please remove all references before deleting.', [
          '@count' => $usage_count,
        ]),
      ];
    }

    // Check if any child terms are in use.
    $child_usage_count = $this->getChildTermUsageCount($tid);
    if ($child_usage_count > 0) {
      return [
        'valid' => FALSE,
        'message' => $this->t('Cannot delete taxonomy term because its child terms are referenced by @count content items. Please remove all references to child terms before deleting the parent.', [
          '@count' => $child_usage_count,
        ]),
      ];
    }

    return [
      'valid' => TRUE,
      'message' => '',
    ];
  }

  /**
   * Get usage count for child terms of a taxonomy term.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return int
   *   Total number of entities using child terms.
   */
  protected function getChildTermUsageCount($tid) {
    $usage_data = $this->usageDetector->getTermUsageWithChildren($tid);
    $count = 0;

    foreach ($usage_data['children'] as $child_data) {
      foreach ($child_data['usage'] as $field_data) {
        $count += count($field_data['entities']);
      }
    }

    return $count;
  }

  /**
   * Check if a taxonomy term can be safely deleted.
   *
   * @param int $tid
   *   The taxonomy term ID.
   *
   * @return bool
   *   TRUE if term can be deleted, FALSE otherwise.
   */
  public function canDeleteTerm($tid) {
    return !$this->usageDetector->isTermInUse($tid);
  }

  /**
   * Get validation errors for multiple terms.
   *
   * @param array $tids
   *   Array of taxonomy term IDs.
   *
   * @return array
   *   Array of validation results keyed by term ID.
   */
  public function validateMultipleTermDeletion(array $tids) {
    $results = [];

    foreach ($tids as $tid) {
      $results[$tid] = $this->validateTermDeletion($tid);
    }

    return $results;
  }

}
