<?php

namespace Drupal\Tests\taxonomy_usage\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\taxonomy\Entity\Term;
use Drupal\node\Entity\NodeType;
use Drupal\node\Entity\Node;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;
use Drupal\media\Entity\MediaType;
use Drupal\media\Entity\Media;

/**
 * Kernel tests for TaxonomyUsageDetector service with real entities.
 *
 * @group taxonomy_usage
 */
class TaxonomyUsageDetectorKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'filter',
    'file',
    'image',
    'node',
    'taxonomy',
    'media',
    'taxonomy_usage',
  ];

  /**
   * The taxonomy usage detector service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDetector
   */
  protected $detector;

  /**
   * The test vocabulary.
   *
   * @var \Drupal\taxonomy\Entity\Vocabulary
   */
  protected $vocabulary;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('taxonomy_term');
    $this->installEntitySchema('file');
    $this->installEntitySchema('media');
    $this->installSchema('node', ['node_access']);
    $this->installSchema('file', ['file_usage']);
    $this->installConfig(['taxonomy_usage', 'field', 'node', 'taxonomy', 'file', 'image']);

    $this->detector = $this->container->get('taxonomy_usage.detector');

    // Create a test vocabulary.
    $this->vocabulary = Vocabulary::create([
      'vid' => 'test_vocab',
      'name' => 'Test Vocabulary',
    ]);
    $this->vocabulary->save();
  }

  /**
   * Test that getSafeTableName returns correct table names.
   */
  public function testGetSafeTableName() {
    // Use reflection to access the protected method.
    $method = new \ReflectionMethod(get_class($this->detector), 'getSafeTableName');
    $method->setAccessible(TRUE);

    // Test data table.
    $result = $method->invoke($this->detector, 'node', 'field_tags', FALSE);
    $this->assertEquals('node__field_tags', $result);

    // Test revision table.
    $result = $method->invoke($this->detector, 'node', 'field_tags', TRUE);
    $this->assertEquals('node_revision__field_tags', $result);
  }

  /**
   * Test term usage detection with nodes.
   */
  public function testTermUsageDetectionWithNodes() {
    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Create a taxonomy reference field.
    FieldStorageConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Tags',
      'settings' => [
        'handler' => 'default',
        'handler_settings' => [
          'target_bundles' => ['test_vocab' => 'test_vocab'],
        ],
      ],
    ])->save();

    // Create a taxonomy term.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Test Term',
    ]);
    $term->save();

    // Initially, term should not be in use.
    $this->assertFalse($this->detector->isTermInUse($term->id()));
    $this->assertEquals(0, $this->detector->getUsageCount($term->id()));

    // Create a node using this term.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Node',
      'field_tags' => ['target_id' => $term->id()],
    ]);
    $node->save();

    // Now term should be in use.
    $this->assertTrue($this->detector->isTermInUse($term->id()));
    $this->assertEquals(1, $this->detector->getUsageCount($term->id()));

    // Test getTermUsageByField.
    $usage = $this->detector->getTermUsageByField($term->id());
    $this->assertNotEmpty($usage);
  }

  /**
   * Test term usage detection with media entities.
   */
  public function testTermUsageDetectionWithMedia() {
    // Create an image field storage for the media source.
    FieldStorageConfig::create([
      'field_name' => 'field_media_image',
      'entity_type' => 'media',
      'type' => 'image',
    ])->save();

    // Create a media type first.
    $media_type = MediaType::create([
      'id' => 'test_media',
      'label' => 'Test Media',
      'source' => 'image',
    ]);
    $media_type->save();

    // Create the image field instance.
    FieldConfig::create([
      'field_name' => 'field_media_image',
      'entity_type' => 'media',
      'bundle' => 'test_media',
      'label' => 'Image',
    ])->save();

    // Set source field configuration.
    $media_type->set('source_configuration', [
      'source_field' => 'field_media_image',
    ]);
    $media_type->save();

    // Create a taxonomy reference field on media.
    FieldStorageConfig::create([
      'field_name' => 'field_blog',
      'entity_type' => 'media',
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_blog',
      'entity_type' => 'media',
      'bundle' => 'test_media',
      'label' => 'Blog',
      'settings' => [
        'handler' => 'default',
        'handler_settings' => [
          'target_bundles' => ['test_vocab' => 'test_vocab'],
        ],
      ],
    ])->save();

    // Create a taxonomy term.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Blog Term',
    ]);
    $term->save();

    // Initially, term should not be in use.
    $this->assertFalse($this->detector->isTermInUse($term->id()));

    // Create a media entity using this term.
    $media = Media::create([
      'bundle' => 'test_media',
      'name' => 'Test Media',
      'field_blog' => ['target_id' => $term->id()],
    ]);
    $media->save();

    // Now term should be in use.
    $this->assertTrue($this->detector->isTermInUse($term->id()));
    $this->assertEquals(1, $this->detector->getUsageCount($term->id()));
  }

  /**
   * Test usage summary by entity type.
   */
  public function testGetUsageSummaryByEntityType() {
    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Create a taxonomy reference field.
    FieldStorageConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Tags',
      'settings' => [
        'handler' => 'default',
        'handler_settings' => [
          'target_bundles' => ['test_vocab' => 'test_vocab'],
        ],
      ],
    ])->save();

    // Create a taxonomy term.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Test Term',
    ]);
    $term->save();

    // Create multiple nodes using this term.
    for ($i = 0; $i < 3; $i++) {
      $node = Node::create([
        'type' => 'article',
        'title' => "Test Node $i",
        'field_tags' => ['target_id' => $term->id()],
      ]);
      $node->save();
    }

    // Get usage summary.
    $summary = $this->detector->getUsageSummaryByEntityType($term->id());
    $this->assertArrayHasKey('node', $summary);
    $this->assertEquals(3, $summary['node']);
  }

  /**
   * Test cache clearing.
   */
  public function testClearTermCache() {
    // Create a term.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Test Term',
    ]);
    $term->save();

    // Call getUsageCount to populate cache.
    $this->detector->getUsageCount($term->id());

    // Clear cache should not throw errors.
    $this->detector->clearTermCache($term->id());

    // Should still be able to get usage count after cache clear.
    $count = $this->detector->getUsageCount($term->id());
    $this->assertEquals(0, $count);
  }

}
