<?php

namespace Drupal\Tests\taxonomy_usage\Unit;

use Drupal\taxonomy_usage\TaxonomyUsageDetector;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Unit tests for TaxonomyUsageDetector service.
 *
 * @group taxonomy_usage
 * @coversDefaultClass \Drupal\taxonomy_usage\TaxonomyUsageDetector
 */
class TaxonomyUsageDetectorTest extends UnitTestCase {

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The mocked entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityFieldManager;

  /**
   * The mocked cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $cache;

  /**
   * The mocked config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The mocked module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The taxonomy usage detector service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDetector
   */
  protected $detector;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->database = $this->createMock(Connection::class);
    $this->entityFieldManager = $this->createMock(EntityFieldManagerInterface::class);
    $this->cache = $this->createMock(CacheBackendInterface::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);

    $this->detector = new TaxonomyUsageDetector(
      $this->entityTypeManager,
      $this->database,
      $this->entityFieldManager,
      $this->cache,
      $this->configFactory,
      $this->moduleHandler
    );
  }

  /**
   * Test getSafeTableName method for data tables.
   *
   * @covers ::getSafeTableName
   */
  public function testGetSafeTableNameForDataTable() {
    // Use reflection to access the protected method.
    $method = new \ReflectionMethod(TaxonomyUsageDetector::class, 'getSafeTableName');
    $method->setAccessible(TRUE);

    // Test standard field.
    $result = $method->invoke($this->detector, 'node', 'field_tags');
    $this->assertEquals('node__field_tags', $result);

    // Test media field.
    $result = $method->invoke($this->detector, 'media', 'field_blog');
    $this->assertEquals('media__field_blog', $result);

    // Test paragraph field.
    $result = $method->invoke($this->detector, 'paragraph', 'field_category');
    $this->assertEquals('paragraph__field_category', $result);
  }

  /**
   * Test getSafeTableName method for revision tables.
   *
   * @covers ::getSafeTableName
   */
  public function testGetSafeTableNameForRevisionTable() {
    // Use reflection to access the protected method.
    $method = new \ReflectionMethod(TaxonomyUsageDetector::class, 'getSafeTableName');
    $method->setAccessible(TRUE);

    // Test standard field revision.
    $result = $method->invoke($this->detector, 'node', 'field_tags', TRUE);
    $this->assertEquals('node_revision__field_tags', $result);

    // Test media field revision.
    $result = $method->invoke($this->detector, 'media', 'field_blog', TRUE);
    $this->assertEquals('media_revision__field_blog', $result);

    // Test paragraph field revision.
    $result = $method->invoke($this->detector, 'paragraph', 'field_category', TRUE);
    $this->assertEquals('paragraph_revision__field_category', $result);
  }

  /**
   * Test that the service can be instantiated.
   *
   * @covers ::__construct
   */
  public function testServiceInstantiation() {
    $this->assertInstanceOf(TaxonomyUsageDetector::class, $this->detector);
  }

  /**
   * Test isParagraphsModuleAvailable method.
   *
   * @covers ::isParagraphsModuleAvailable
   */
  public function testIsParagraphsModuleAvailable() {
    // Use reflection to access the protected method.
    $method = new \ReflectionMethod(TaxonomyUsageDetector::class, 'isParagraphsModuleAvailable');
    $method->setAccessible(TRUE);

    // Test when paragraphs is enabled.
    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('paragraphs')
      ->willReturn(TRUE);

    $result = $method->invoke($this->detector);
    $this->assertTrue($result);
  }

  /**
   * Test tableExists method.
   *
   * @covers ::tableExists
   */
  public function testTableExists() {
    // Use reflection to access the protected method.
    $method = new \ReflectionMethod(TaxonomyUsageDetector::class, 'tableExists');
    $method->setAccessible(TRUE);

    $schema = $this->getMockBuilder('stdClass')
      ->addMethods(['tableExists'])
      ->getMock();

    $schema->expects($this->once())
      ->method('tableExists')
      ->with('node__field_tags')
      ->willReturn(TRUE);

    $this->database->expects($this->once())
      ->method('schema')
      ->willReturn($schema);

    $result = $method->invoke($this->detector, 'node__field_tags');
    $this->assertTrue($result);
  }

}
