<?php

namespace Drupal\Tests\taxonomy_usage\Unit;

use Drupal\taxonomy_usage\TaxonomyUsageDisplay;
use Drupal\taxonomy_usage\TaxonomyUsageDetector;
use Drupal\Tests\UnitTestCase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Unit tests for TaxonomyUsageDisplay service.
 *
 * @group taxonomy_usage
 * @coversDefaultClass \Drupal\taxonomy_usage\TaxonomyUsageDisplay
 */
class TaxonomyUsageDisplayTest extends UnitTestCase {

  /**
   * The mocked taxonomy usage detector.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDetector|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $usageDetector;

  /**
   * The mocked string translation.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $stringTranslation;

  /**
   * The mocked module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $moduleHandler;

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The mocked database connection.
   *
   * @var \Drupal\Core\Database\Connection|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $database;

  /**
   * The taxonomy usage display service.
   *
   * @var \Drupal\taxonomy_usage\TaxonomyUsageDisplay
   */
  protected $display;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->usageDetector = $this->createMock(TaxonomyUsageDetector::class);
    $this->stringTranslation = $this->createMock(TranslationInterface::class);
    $this->moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->database = $this->createMock(Connection::class);

    $this->display = new TaxonomyUsageDisplay(
      $this->usageDetector,
      $this->stringTranslation,
      $this->moduleHandler,
      $this->entityTypeManager,
      $this->database
    );
  }

  /**
   * Test that the service can be instantiated with all 5 arguments.
   *
   * This test verifies the fix for the ArgumentCountError bug where
   * only 3 arguments were being passed instead of 5.
   *
   * @covers ::__construct
   */
  public function testServiceInstantiationWithAllArguments() {
    $this->assertInstanceOf(TaxonomyUsageDisplay::class, $this->display);
  }

  /**
   * Test that constructor throws error with insufficient arguments.
   *
   * @covers ::__construct
   */
  public function testServiceInstantiationWithInsufficientArguments() {
    $this->expectException(\ArgumentCountError::class);

    // Try to instantiate with only 3 arguments (should fail).
    new TaxonomyUsageDisplay(
      $this->usageDetector,
      $this->stringTranslation,
      $this->moduleHandler
    );
  }

  /**
   * Test getUsageCount method delegates to detector.
   *
   * @covers ::getUsageCount
   */
  public function testGetUsageCount() {
    $tid = 123;
    $expected_count = 42;

    $this->usageDetector->expects($this->once())
      ->method('getUsageCount')
      ->with($tid)
      ->willReturn($expected_count);

    $result = $this->display->getUsageCount($tid);
    $this->assertEquals($expected_count, $result);
  }

  /**
   * Test isTermInUse method delegates to detector.
   *
   * @covers ::isTermInUse
   */
  public function testIsTermInUse() {
    $tid = 123;

    $this->usageDetector->expects($this->once())
      ->method('isTermInUse')
      ->with($tid)
      ->willReturn(TRUE);

    $result = $this->display->isTermInUse($tid);
    $this->assertTrue($result);
  }

  /**
   * Test isParagraphsModuleAvailable method.
   *
   * @covers ::isParagraphsModuleAvailable
   */
  public function testIsParagraphsModuleAvailable() {
    // Use reflection to access the protected method.
    $method = new \ReflectionMethod(TaxonomyUsageDisplay::class, 'isParagraphsModuleAvailable');
    $method->setAccessible(TRUE);

    $this->moduleHandler->expects($this->once())
      ->method('moduleExists')
      ->with('paragraphs')
      ->willReturn(TRUE);

    $result = $method->invoke($this->display);
    $this->assertTrue($result);
  }

}
