<?php

namespace Drupal\telephone_advanced;

use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumber;
use libphonenumber\PhoneNumberType;
use libphonenumber\PhoneNumberUtil;

/**
 * Provides the telephone parser.
 */
class TelephoneParser implements TelephoneParserInterface {

  /**
   * The cached results.
   *
   * @var array
   */
  protected array $results = [];

  /**
   * {@inheritdoc}
   */
  public function parse(string $number, ?string $default_country = NULL): PhoneNumber {
    $key = $default_country . '|' . $number;

    if (!isset($this->results[$key])) {
      $this->results[$key] = PhoneNumberUtil::getInstance()
        ->parse($number, $default_country);
    }

    return $this->results[$key];
  }

  /**
   * {@inheritdoc}
   */
  public function getCountry(PhoneNumber|string $number, ?string $default_country = NULL): ?string {
    if (!$number instanceof PhoneNumber) {
      try {
        $number = $this->parse($number, $default_country);
      }
      catch (NumberParseException $ex) {
        return NULL;
      }
    }

    return PhoneNumberUtil::getInstance()->getRegionCodeForNumber($number);
  }

  /**
   * {@inheritdoc}
   */
  public function getType(PhoneNumber|string $number, ?string $default_country = NULL): ?int {
    if (!$number instanceof PhoneNumber) {
      try {
        $number = $this->parse($number, $default_country);
      }
      catch (NumberParseException $ex) {
        return NULL;
      }
    }

    $type = PhoneNumberUtil::getInstance()->getNumberType($number);

    if ($type === PhoneNumberType::UNKNOWN) {
      return NULL;
    }

    return TelephoneTypes::getId($type);
  }

}
