<?php

namespace Drupal\telephone_advanced;

use libphonenumber\PhoneNumber;

/**
 * Interface for the telephone parser service.
 */
interface TelephoneParserInterface {

  /**
   * Parse a telephone number.
   *
   * @param string $number
   *   The telephone number.
   * @param string|null $default_country
   *   The default country code, leave NULL if the number is guaranteed
   *   to start with a country calling code.
   *
   * @return \libphonenumber\PhoneNumber
   *   The phone number object.
   *
   * @throws \libphonenumber\NumberParseException
   */
  public function parse(string $number, ?string $default_country = NULL): PhoneNumber;

  /**
   * Get the country of a telephone number.
   *
   * @param \libphonenumber\PhoneNumber|string $number
   *   The telephone number.
   * @param string|null $default_country
   *   The default country code. Can be NULL when $number is a PhoneNumber object
   *   or if it's guaranteed to start with a country calling code.
   *
   * @return string|null
   *   The country of the telephone number or NULL if unknown.
   */
  public function getCountry(PhoneNumber|string $number, ?string $default_country = NULL): ?string;

  /**
   * Get the telephone number type.
   *
   * @param string|\libphonenumber\PhoneNumber $number
   *   The telephone number.
   * @param string|null $default_country
   *   The default country code. Can be NULL when $number is a PhoneNumber object
   *   or if it's guaranteed to start with a country calling code.
   *
   * @return int|null
   *   The telephone number type ID or NULL if unknown.
   */
  public function getType(PhoneNumber|string $number, ?string $default_country = NULL): ?int;

}
