<?php

namespace Drupal\telephone_advanced;

use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumber;
use libphonenumber\PhoneNumberType;

/**
 * Provides the telephone validator.
 */
class TelephoneValidator implements TelephoneValidatorInterface {

  /**
   * Class constructor.
   *
   * @param \Drupal\telephone_advanced\TelephoneParserInterface $telephoneParser
   *   The telephone parser.
   */
  public function __construct(
    protected readonly TelephoneParserInterface $telephoneParser,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function isValid(string $number, ?string $default_country = NULL): bool {
    try {
      $this->telephoneParser->parse($number, $default_country);
    }
    catch (NumberParseException $ex) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function isFromCountry(PhoneNumber|string $number, string|array $country, ?string $default_country = NULL): bool {
    $number_country = $this->telephoneParser->getCountry($number, $default_country);

    if ($number_country === NULL) {
      return FALSE;
    }

    return in_array($number_country, (array) $country, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function isOfType(PhoneNumber|string $number, PhoneNumberType|int|array $type, bool $strict = FALSE, ?string $default_country = NULL): bool {
    $number_type = $this->telephoneParser->getType($number, $default_country);

    if ($number_type === NULL) {
      return FALSE;
    }

    $type = array_map([TelephoneTypes::class, 'getId'], (array) $type);

    if (!$strict) {
      if (in_array(TelephoneTypes::getId(PhoneNumberType::FIXED_LINE), $type, TRUE)) {
        $type[] = TelephoneTypes::getId(PhoneNumberType::FIXED_LINE_OR_MOBILE);
      }

      if (in_array(TelephoneTypes::getId(PhoneNumberType::MOBILE), $type, TRUE)) {
        $type[] = TelephoneTypes::getId(PhoneNumberType::FIXED_LINE_OR_MOBILE);
      }
    }

    return in_array($number_type, $type, TRUE);
  }

}
