<?php

namespace Drupal\temp_admin_login\Controller;

use Drupal\Component\Utility\Random;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\Core\Url;
use Drupal\user\Entity\User;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for generating and handling temporary admin login links.
 */
class TempAdminLoginController extends ControllerBase {

  /**
   * Generates a temporary login link for the specified role.
   *
   * @param string $role
   *   The role to associate with the temporary login link.
   * @param int $expiration_time
   *   The time (in seconds) until the link expires.
   *
   * @return array
   *   A render array containing the temporary login link.
   */
  public function generateLink($role, $expiration_time) {
    $random = new Random();
    $token = $random->string(32);

    // Store the token, timestamp, role, and expiration time in the database.
    $connection = Database::getConnection();
    $connection->insert('temp_admin_login')
      ->fields([
        'token' => $token,
        'created' => \Drupal::time()->getRequestTime(),
        'role' => $role,
        'expiration' => $expiration_time,
      ])
      ->execute();

    $url = Url::fromRoute('temp_admin_login.login', ['token' => $token], ['absolute' => TRUE])->toString();
    \Drupal::logger('temp_admin_login')->notice('Generated URL: @url', ['@url' => $url]);

    return [
      '#markup' => $this->t('Temporary login link: <a href="@url">@url</a>', ['@url' => $url]),
    ];
  }

  /**
   * Handles login based on the provided token.
   *
   * @param string $token
   *   The token used to authenticate the login attempt.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Symfony\Component\HttpFoundation\Response
   *   A redirect response to the user page on successful login,
   *   or a response with an error message.
   */
  public function login($token) {
    $connection = Database::getConnection();
    $query = $connection->select('temp_admin_login', 'tal')
      ->fields('tal', ['created', 'role', 'expiration'])
      ->condition('token', $token)
      ->execute()
      ->fetchAssoc();

    if ($query) {
      $created_time = $query['created'];
      $current_time = \Drupal::time()->getRequestTime();
      $expiration_time = $query['expiration'];

      if (($current_time - $created_time) < $expiration_time) {
        $admin_user = User::load(1);
        user_login_finalize($admin_user);

        return new RedirectResponse(Url::fromRoute('user.page')->toString());
      }
      else {
        return new Response($this->t('The login link has expired.'), 403);
      }
    }
    else {
      return new Response($this->t('Invalid login link.'), 403);
    }
  }

}
