<?php

namespace Drupal\temp_admin_login\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\Role;

/**
 * Provides a form for generating temporary admin login links.
 */
class TempAdminLoginForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'temp_admin_login_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get available roles.
    $roles = Role::loadMultiple();
    $role_options = [];
    foreach ($roles as $role) {
      $role_options[$role->id()] = $role->label();
    }

    $form['role'] = [
      '#type' => 'select',
      '#title' => $this->t('Role'),
      '#options' => $role_options,
      '#required' => TRUE,
      '#description' => $this->t('Select the role for the temporary login.'),
    ];

    $form['expiration_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Expiration Time (h:m:s)'),
      '#required' => TRUE,
      '#default_value' => '01:00:00',
      '#description' => $this->t('Enter the expiration time for the link in h:m:s format.'),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate Link'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $role = $form_state->getValue('role');
    $expiration_time = $form_state->getValue('expiration_time');

    // Convert h:m:s format to seconds.
    [$hours, $minutes, $seconds] = explode(':', $expiration_time);
    $expiration_time_in_seconds = ($hours * 3600) + ($minutes * 60) + $seconds;

    // Redirect to the controller to generate the link.
    $form_state->setRedirect('temp_admin_login.generate_link', [
      'role' => $role,
      'expiration_time' => $expiration_time_in_seconds,
    ]);
  }

}
