# Testing Guide for Term Delete Protection

## Prerequisites

- Drupal 10+ installed
- PHPUnit configured (comes with Drupal core)
- For DDEV users: DDEV environment running

## Environment Setup

### Configure PHPUnit (if not already done)

Copy the core phpunit configuration:
```bash
cp web/core/phpunit.xml.dist web/core/phpunit.xml
```

Edit `web/core/phpunit.xml` and update:
```xml
<env name="SIMPLETEST_BASE_URL" value="http://d10dev.ddev.site"/>
<env name="SIMPLETEST_DB" value="mysql://db:db@db/db"/>
```

For DDEV, the database connection is typically:
- Driver: mysql
- Host: db
- Database: db
- Username: db
- Password: db

## Running Tests

### All Tests for the Module

```bash
# Using DDEV
ddev exec vendor/bin/phpunit -c web/core --group term_delete_protection web/modules/custom/term_delete_protection

# Without DDEV
vendor/bin/phpunit -c web/core --group term_delete_protection web/modules/custom/term_delete_protection
```

### Kernel Tests Only

```bash
# Using DDEV
ddev exec vendor/bin/phpunit -c web/core --group term_delete_protection web/modules/custom/term_delete_protection/tests/src/Kernel

# Without DDEV
vendor/bin/phpunit -c web/core --group term_delete_protection web/modules/custom/term_delete_protection/tests/src/Kernel
```

### Functional Tests Only

```bash
# Using DDEV
ddev exec vendor/bin/phpunit -c web/core --group term_delete_protection web/modules/custom/term_delete_protection/tests/src/Functional

# Without DDEV
vendor/bin/phpunit -c web/core --group term_delete_protection web/modules/custom/term_delete_protection/tests/src/Functional
```

### Run a Specific Test Class

```bash
ddev exec vendor/bin/phpunit -c web/core web/modules/custom/term_delete_protection/tests/src/Kernel/TermReferenceCheckerTest.php
```

### Run a Specific Test Method

```bash
ddev exec vendor/bin/phpunit -c web/core --filter testReferencedTerm web/modules/custom/term_delete_protection/tests/src/Kernel/TermReferenceCheckerTest.php
```

## Test Output Options

### Verbose Output

```bash
ddev exec vendor/bin/phpunit -c web/core --verbose --group term_delete_protection web/modules/custom/term_delete_protection
```

### Show Test Coverage

```bash
ddev exec vendor/bin/phpunit -c web/core --coverage-html coverage --group term_delete_protection web/modules/custom/term_delete_protection
```

This generates an HTML coverage report in the `coverage/` directory.

### Debug Output

```bash
ddev exec vendor/bin/phpunit -c web/core --debug --group term_delete_protection web/modules/custom/term_delete_protection
```

## Common Issues

### Issue: SIMPLETEST_BASE_URL not set

**Error:** `RuntimeException: You must provide a SIMPLETEST_BASE_URL environment variable`

**Solution:** Set the base URL in phpunit.xml or via environment variable:
```bash
export SIMPLETEST_BASE_URL=http://d10dev.ddev.site
```

### Issue: Database connection fails

**Error:** `Could not connect to the database`

**Solution:** Verify your SIMPLETEST_DB setting in phpunit.xml matches your environment.

For DDEV:
```xml
<env name="SIMPLETEST_DB" value="mysql://db:db@db/db"/>
```

### Issue: Tests fail with "No such file or directory"

**Solution:** Make sure you're running tests from the Drupal root directory.

### Issue: Functional tests fail with browser errors

**Solution:** Functional tests require a web server. With DDEV, the container provides this automatically. Make sure DDEV is running.

## Code Quality Checks

### PHP_CodeSniffer

Check coding standards:
```bash
ddev exec vendor/bin/phpcs --standard=Drupal,DrupalPractice web/modules/custom/term_delete_protection
```

Auto-fix coding standards issues:
```bash
ddev exec vendor/bin/phpcbf --standard=Drupal,DrupalPractice web/modules/custom/term_delete_protection
```

### PHPStan (Static Analysis)

If you have PHPStan installed:
```bash
ddev exec vendor/bin/phpstan analyze web/modules/custom/term_delete_protection
```

## Drupal.org GitLab CI

When you push code to Drupal.org, the `.gitlab-ci.yml` file automatically:

1. Runs PHP_CodeSniffer for code quality
1. Runs all PHPUnit tests
1. Generates test reports
1. Archives browser test output (for functional test debugging)

View test results on your project's pipelines page:
`https://git.drupalcode.org/project/term_delete_protection/-/pipelines`

## Writing New Tests

### For Kernel Tests

1. Extend `KernelTestBase`
1. Set up test entities in `setUp()`
1. Test business logic and services
1. No browser/UI testing

### For Functional Tests

1. Extend `BrowserTestBase`
1. Create admin user in `setUp()`
1. Use `$this->drupalGet()` to visit pages
1. Use `$this->assertSession()` for assertions
1. Test full user workflows

### Test Naming Conventions

- Test class: `*Test.php` (e.g., `TermReferenceCheckerTest.php`)
- Test methods: `test*()` (e.g., `testReferencedTerm()`)
- Use descriptive names that explain what is being tested

### Documentation

Always add a docblock to test classes and methods:

```php
/**
 * Tests term reference checking for commerce products.
 *
 * @group term_delete_protection
 */
public function testCommerceProductReferences() {
  // Test implementation
}
```

## Resources

- [Drupal PHPUnit Documentation](https://www.drupal.org/docs/automated-testing/phpunit-in-drupal)
- [Writing PHPUnit Tests in Drupal](https://www.drupal.org/docs/automated-testing/phpunit-in-drupal/running-phpunit-tests)
- [BrowserTestBase](https://api.drupal.org/api/drupal/core!tests!Drupal!Tests!BrowserTestBase.php/class/BrowserTestBase)
- [KernelTestBase](https://api.drupal.org/api/drupal/core!tests!Drupal!KernelTests!KernelTestBase.php/class/KernelTestBase)
