<?php

namespace Drupal\term_delete_protection\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\term_delete_protection\Service\TermReferenceChecker;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Event subscriber for term delete protection.
 */
class TermDeleteProtectionSubscriber implements EventSubscriberInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The term reference checker service.
   *
   * @var \Drupal\term_delete_protection\Service\TermReferenceChecker
   */
  protected $termReferenceChecker;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new TermDeleteProtectionSubscriber.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory service.
   * @param \Drupal\term_delete_protection\Service\TermReferenceChecker $term_reference_checker
   *   The term reference checker service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    AccountProxyInterface $current_user,
    MessengerInterface $messenger,
    LoggerChannelFactoryInterface $logger_factory,
    TermReferenceChecker $term_reference_checker,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->currentUser = $current_user;
    $this->messenger = $messenger;
    $this->loggerFactory = $logger_factory;
    $this->termReferenceChecker = $term_reference_checker;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['onRequest', 28],
    ];
  }

  /**
   * Handles the request event.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onRequest(RequestEvent $event) {
    $route = $event->getRequest()->attributes->get('_route');

    if ($route === 'entity.taxonomy_term.delete_form') {
      $term_route = $event->getRequest()->attributes->get('taxonomy_term');

      if ($term_route) {
        try {
          // Get term ID whether term_route is object or integer.
          $term_id = is_object($term_route) ? $term_route->id() : $term_route;
          $term = is_object($term_route) ? $term_route : $this->entityTypeManager->getStorage('taxonomy_term')->load($term_id);

          if (!$term) {
            return;
          }

          $vocabulary_id = $term->bundle();

          // Check if protection is enabled for this vocabulary.
          $config = $this->configFactory->get('term_delete_protection.settings');
          $vocabularies = $config->get('vocabularies') ?: [];

          $is_protected = FALSE;
          foreach ($vocabularies as $vocab_settings) {
            if ($vocab_settings['vocabulary_id'] === $vocabulary_id && !empty($vocab_settings['protected_entity_types'])) {
              $is_protected = TRUE;
              break;
            }
          }

          if (!$is_protected) {
            // Protection not enabled for this vocabulary.
            return;
          }

          // Use our service to check term references.
          $result = $this->termReferenceChecker->checkTermReferences($term);

          if ($result['has_references']) {
            $this->messenger->addError($result['reason']);
            $event->setResponse(new RedirectResponse($term->toUrl()->toString()));
          }
        }
        catch (\Exception $e) {
          $this->loggerFactory->get('term_delete_protection')->error($e->getMessage());
        }
      }
    }
  }

}
