<?php

namespace Drupal\term_delete_protection\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\taxonomy\TermInterface;

/**
 * Service for checking term references in products.
 */
class TermReferenceChecker {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new TermReferenceChecker.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    EntityFieldManagerInterface $entity_field_manager,
    ModuleHandlerInterface $module_handler,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->entityFieldManager = $entity_field_manager;
    $this->moduleHandler = $module_handler;
    $this->configFactory = $config_factory;
  }

  /**
   * Check if a term is referenced by entities (nodes, products, etc.).
   *
   * @param \Drupal\taxonomy\TermInterface $term
   *   The term to check.
   *
   * @return array
   *   An array with keys:
   *   - 'has_references': TRUE if any reference check passes, FALSE otherwise.
   *   - 'reason': A string indicating why the term can't be deleted, empty if no references.
   */
  public function checkTermReferences(TermInterface $term): array {
    $term_id = $term->id();
    $vocabulary_id = $term->bundle();

    if (!$term_id) {
      return [
        'has_references' => FALSE,
        'reason' => '',
      ];
    }

    // Check if term itself is directly referenced by entities.
    if ($this->isTermReferencedByEntity($term_id, $vocabulary_id)) {
      return [
        'has_references' => TRUE,
        'reason' => 'This term (' . $term->label() . ') cannot be deleted because it is referenced by content.',
      ];
    }

    // Check if any descendant terms are referenced.
    if ($this->hasDescendantTermsReferencedByEntities($term_id, $vocabulary_id)) {
      return [
        'has_references' => TRUE,
        'reason' => 'This term (' . $term->label() . ') cannot be deleted because one or more of its descendant terms are referenced by content.',
      ];
    }

    return [
      'has_references' => FALSE,
      'reason' => '',
    ];
  }

  /**
   * Checks if a term is referenced by any protected entities.
   *
   * @param int $term_id
   *   The term ID to check.
   * @param string $vocabulary_id
   *   The vocabulary ID to get protected entity types.
   *
   * @return bool
   *   TRUE if the term is referenced, FALSE otherwise.
   */
  public function isTermReferencedByEntity(int $term_id, string $vocabulary_id): bool {
    $protected_entity_types = $this->getProtectedEntityTypes($vocabulary_id);

    if (empty($protected_entity_types)) {
      return FALSE;
    }

    foreach ($protected_entity_types as $entity_type) {
      if ($this->isTermReferencedByEntityType($term_id, $entity_type)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Get protected entity types for a vocabulary.
   *
   * @param string $vocabulary_id
   *   The vocabulary machine name.
   *
   * @return array
   *   Array of protected entity type IDs.
   */
  protected function getProtectedEntityTypes(string $vocabulary_id): array {
    $config = $this->configFactory->get('term_delete_protection.settings');
    $vocabularies = $config->get('vocabularies') ?: [];

    foreach ($vocabularies as $vocab_settings) {
      if ($vocab_settings['vocabulary_id'] === $vocabulary_id) {
        return $vocab_settings['protected_entity_types'] ?? [];
      }
    }

    return [];
  }

  /**
   * Checks if a term is referenced by a specific entity type.
   *
   * @param int $term_id
   *   The term ID to check.
   * @param string $entity_type_id
   *   The entity type ID (e.g., 'node', 'commerce_product', 'paragraph').
   *
   * @return bool
   *   TRUE if the term is referenced, FALSE otherwise.
   */
  protected function isTermReferencedByEntityType(int $term_id, string $entity_type_id): bool {
    // Get all taxonomy reference fields for this entity type.
    $field_map = $this->entityFieldManager
      ->getFieldMapByFieldType('entity_reference');

    if (!isset($field_map[$entity_type_id])) {
      return FALSE;
    }

    // Check each taxonomy reference field.
    foreach ($field_map[$entity_type_id] as $field_name => $field_info) {
      // Get field config for each bundle.
      foreach ($field_info['bundles'] as $bundle) {
        $field_config = $this->entityTypeManager
          ->getStorage('field_config')
          ->load("{$entity_type_id}.{$bundle}.{$field_name}");

        if (!$field_config) {
          continue;
        }

        // Check if this field references taxonomy terms.
        $settings = $field_config->getSettings();
        if (isset($settings['handler']) && $settings['handler'] === 'default:taxonomy_term') {
          // Build query conditions based on entity type.
          $query = $this->entityTypeManager
            ->getStorage($entity_type_id)
            ->getQuery()
            ->condition($field_name, $term_id)
            ->accessCheck(TRUE)
            ->range(0, 1);

          // Add status condition for entity types that have it.
          if (in_array($entity_type_id, ['node', 'commerce_product'])) {
            $query->condition('status', 1);
          }

          $results = $query->execute();
          if (!empty($results)) {
            return TRUE;
          }
        }
      }
    }

    return FALSE;
  }

  /**
   * Checks if a term has descendant terms referenced by entities.
   *
   * This method recursively checks all children, grandchildren, etc.
   *
   * @param int $term_id
   *   The term ID to check.
   * @param string $vocabulary_id
   *   The vocabulary ID.
   * @param array $checked_terms
   *   Array of already checked term IDs to prevent infinite loops.
   *
   * @return bool
   *   TRUE if any descendant is referenced, FALSE otherwise.
   */
  protected function hasDescendantTermsReferencedByEntities(int $term_id, string $vocabulary_id, array &$checked_terms = []): bool {
    // Prevent infinite loops.
    if (in_array($term_id, $checked_terms)) {
      return FALSE;
    }
    $checked_terms[] = $term_id;

    // Load all children terms.
    $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $term = $term_storage->load($term_id);

    if (!$term) {
      return FALSE;
    }

    $child_terms = $term_storage->loadTree(
      $term->bundle(),
      $term_id,
      NULL,
      FALSE
    );

    if (empty($child_terms)) {
      return FALSE;
    }

    // Check each child term.
    foreach ($child_terms as $child_term) {
      $child_term_id = $child_term->tid;

      // Check if this child is directly referenced.
      if ($this->isTermReferencedByEntity($child_term_id, $vocabulary_id)) {
        return TRUE;
      }

      // Recursively check descendants of this child.
      if ($this->hasDescendantTermsReferencedByEntities($child_term_id, $vocabulary_id, $checked_terms)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Get all related entities grouped by entity type.
   *
   * @param int $term_id
   *   The term ID to check.
   * @param string $vocabulary_id
   *   The vocabulary ID.
   *
   * @return array
   *   Array of entities grouped by entity type.
   */
  public function getAllRelatedEntities(int $term_id, string $vocabulary_id): array {
    $protected_entity_types = $this->getProtectedEntityTypes($vocabulary_id);
    $all_entities = [];

    foreach ($protected_entity_types as $entity_type) {
      $entities = $this->getAllRelatedEntitiesByType($term_id, $entity_type);
      if (!empty($entities)) {
        $all_entities[$entity_type] = $entities;
      }
    }

    return $all_entities;
  }

  /**
   * Get entities of a specific type that reference a term or descendants.
   *
   * @param int $term_id
   *   The term ID to check.
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   Array with all the entity data for the term and descendants.
   */
  protected function getAllRelatedEntitiesByType(int $term_id, string $entity_type_id): array {
    $term_storage = $this->entityTypeManager->getStorage('taxonomy_term');
    $term = $term_storage->load($term_id);

    if (!$term) {
      return [];
    }

    $entity_data = [];

    // Check the term itself.
    $entities = $this->getEntitiesForTerm($term_id, $term->label(), $entity_type_id);
    if (!empty($entities)) {
      $entity_data[$term_id] = $entities;
    }

    // Get all descendant terms using loadTree.
    $descendant_terms = $term_storage->loadTree(
      $term->bundle(),
      $term_id,
      NULL,
      FALSE
    );

    // Check each descendant term.
    foreach ($descendant_terms as $descendant_term) {
      $entities = $this->getEntitiesForTerm($descendant_term->tid, $descendant_term->name, $entity_type_id);
      if (!empty($entities)) {
        $entity_data[$descendant_term->tid] = $entities;
      }
    }

    return $entity_data;
  }

  /**
   * Get entities that reference a specific term.
   *
   * @param int $term_id
   *   The term ID.
   * @param string $term_name
   *   The term name/label.
   * @param string $entity_type_id
   *   The entity type ID.
   *
   * @return array
   *   Array with term name and associated entities.
   */
  protected function getEntitiesForTerm(int $term_id, string $term_name, string $entity_type_id): array {
    $data = [
      'term_name' => $term_name,
      'entities' => [],
    ];

    // Get all taxonomy reference fields for this entity type.
    $field_map = $this->entityFieldManager
      ->getFieldMapByFieldType('entity_reference');

    if (!isset($field_map[$entity_type_id])) {
      return [];
    }

    // Check each taxonomy reference field.
    foreach ($field_map[$entity_type_id] as $field_name => $field_info) {
      // Get field config for each bundle.
      foreach ($field_info['bundles'] as $bundle) {
        $field_config = $this->entityTypeManager
          ->getStorage('field_config')
          ->load("{$entity_type_id}.{$bundle}.{$field_name}");

        if (!$field_config) {
          continue;
        }

        // Check if this field references taxonomy terms.
        $settings = $field_config->getSettings();
        if (isset($settings['handler']) && $settings['handler'] === 'default:taxonomy_term') {
          // Query entities with this field referencing our term.
          $query = $this->entityTypeManager
            ->getStorage($entity_type_id)
            ->getQuery()
            ->condition($field_name, $term_id)
            ->accessCheck(TRUE)
            ->range(0, 5)
            ->sort('created', 'DESC');

          // Add status condition for entity types that have it.
          if (in_array($entity_type_id, ['node', 'commerce_product'])) {
            $query->condition('status', 1);
          }

          $entity_ids = $query->execute();

          if (!empty($entity_ids)) {
            $entities = $this->entityTypeManager
              ->getStorage($entity_type_id)
              ->loadMultiple($entity_ids);

            foreach ($entities as $entity) {
              // Special handling for paragraphs - get parent entity.
              if ($entity_type_id === 'paragraph') {
                $parent_entity = $this->getParagraphParentEntity($entity);
                if ($parent_entity) {
                  $data['entities'][] = [
                    'label' => $entity->label() . ' (' . $entity->bundle() . ')',
                    'id' => $parent_entity->id(),
                    'type' => $parent_entity->bundle(),
                    'entity_type' => $parent_entity->getEntityTypeId(),
                    'paragraph_label' => $entity->label(),
                  ];
                }
              }
              else {
                $data['entities'][] = [
                  'label' => $entity->label(),
                  'id' => $entity->id(),
                  'type' => $entity->bundle(),
                  'entity_type' => $entity_type_id,
                ];
              }
            }
          }
        }
      }
    }

    return !empty($data['entities']) ? $data : [];
  }

  /**
   * Get the parent entity of a paragraph.
   *
   * @param \Drupal\Core\Entity\EntityInterface $paragraph
   *   The paragraph entity.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The parent entity or NULL if not found.
   */
  protected function getParagraphParentEntity($paragraph) {
    if (!$paragraph->hasField('parent_id') || !$paragraph->hasField('parent_type')) {
      return NULL;
    }

    $parent_type = $paragraph->get('parent_type')->value;
    $parent_id = $paragraph->get('parent_id')->value;

    if (!$parent_type || !$parent_id) {
      return NULL;
    }

    try {
      $parent_entity = $this->entityTypeManager
        ->getStorage($parent_type)
        ->load($parent_id);
      return $parent_entity;
    }
    catch (\Exception $e) {
      return NULL;
    }
  }

}
