<?php

namespace Drupal\Tests\term_delete_protection\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Tests the Term Delete Protection module's UI functionality.
 *
 * @group term_delete_protection
 */
class TermDeleteProtectionTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'taxonomy',
    'field',
    'term_delete_protection',
  ];

  /**
   * A test vocabulary.
   *
   * @var \Drupal\taxonomy\VocabularyInterface
   */
  protected $vocabulary;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a test vocabulary.
    $this->vocabulary = Vocabulary::create([
      'vid' => 'test_vocab',
      'name' => 'Test Vocabulary',
    ]);
    $this->vocabulary->save();

    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Create a taxonomy reference field.
    FieldStorageConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Tags',
      'settings' => [
        'handler' => 'default:taxonomy_term',
        'handler_settings' => [
          'target_bundles' => [
            'test_vocab' => 'test_vocab',
          ],
        ],
      ],
    ])->save();

    // Create a user with permission to manage taxonomy.
    $admin_user = $this->drupalCreateUser([
      'administer taxonomy',
      'delete terms in test_vocab',
      'edit terms in test_vocab',
      'administer nodes',
      'create article content',
    ]);
    $this->drupalLogin($admin_user);
  }

  /**
   * Tests vocabulary configuration form.
   */
  public function testVocabularyConfigurationForm() {
    // Visit vocabulary edit form.
    $this->drupalGet('admin/structure/taxonomy/manage/test_vocab');

    // Check that Term Delete Protection fieldset exists.
    $this->assertSession()->pageTextContains('Term Delete Protection');
    $this->assertSession()->fieldExists('protect_nodes');

    // Enable protection for nodes.
    $edit = [
      'protect_nodes' => TRUE,
    ];
    $this->submitForm($edit, 'Save');

    // Verify configuration was saved.
    $config = $this->config('term_delete_protection.settings');
    $vocabularies = $config->get('vocabularies');
    $this->assertNotEmpty($vocabularies, 'Vocabularies configuration should not be empty.');

    $found = FALSE;
    foreach ($vocabularies as $vocab) {
      if ($vocab['vocabulary_id'] === 'test_vocab') {
        $this->assertContains('node', $vocab['protected_entity_types']);
        $found = TRUE;
        break;
      }
    }
    $this->assertTrue($found, 'Protection configuration should be saved for test_vocab.');
  }

  /**
   * Tests that delete operation is removed for referenced terms.
   */
  public function testDeleteOperationRemoval() {
    // Create a term.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Protected Term',
    ]);
    $term->save();

    // Create a node referencing the term.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
      'field_tags' => [
        ['target_id' => $term->id()],
      ],
    ]);
    $node->save();

    // Enable protection.
    $config = \Drupal::configFactory()->getEditable('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    // Visit taxonomy overview page.
    $this->drupalGet('admin/structure/taxonomy/manage/test_vocab/overview');

    // The delete link should not be present for the protected term.
    $this->assertSession()->pageTextContains('Protected Term');
    // Note: Delete link removal is handled by hook_entity_operation_alter.
    // The exact assertion depends on how the page renders operations.
  }

  /**
   * Tests deletion blocking via delete form.
   */
  public function testDeletionBlocking() {
    // Create a term.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Protected Term',
    ]);
    $term->save();

    // Create a node referencing the term.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
      'field_tags' => [
        ['target_id' => $term->id()],
      ],
    ]);
    $node->save();

    // Enable protection.
    $config = \Drupal::configFactory()->getEditable('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    // Try to access the delete form.
    $this->drupalGet('taxonomy/term/' . $term->id() . '/delete');

    // Should be redirected back to term page with error message.
    $this->assertSession()->pageTextContains('cannot be deleted');
    $this->assertSession()->addressEquals('taxonomy/term/' . $term->id());
  }

  /**
   * Tests warning message on term edit form.
   */
  public function testTermEditWarning() {
    // Create a term.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Protected Term',
    ]);
    $term->save();

    // Create a node referencing the term.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
      'field_tags' => [
        ['target_id' => $term->id()],
      ],
    ]);
    $node->save();

    // Enable protection.
    $config = \Drupal::configFactory()->getEditable('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    // Visit term edit form.
    $this->drupalGet('taxonomy/term/' . $term->id() . '/edit');

    // Should see warning message.
    $this->assertSession()->pageTextContains('cannot be deleted');
    $this->assertSession()->pageTextContains('Content (Nodes)');
    $this->assertSession()->linkExists('Test Article');

    // Delete button should be removed.
    $this->assertSession()->buttonNotExists('Delete');
  }

  /**
   * Tests that unreferenced terms can still be deleted.
   */
  public function testUnreferencedTermDeletion() {
    // Create a term without any references.
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Unreferenced Term',
    ]);
    $term->save();

    // Enable protection.
    $config = \Drupal::configFactory()->getEditable('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    // Visit term edit form.
    $this->drupalGet('taxonomy/term/' . $term->id() . '/edit');

    // Should NOT see warning message.
    $this->assertSession()->pageTextNotContains('cannot be deleted');

    // Delete option should be available (either as button or link).
    // In Drupal, delete is typically available, just check we can access delete form.
    $this->drupalGet('taxonomy/term/' . $term->id() . '/delete');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure you want to delete');
  }

}
