<?php

namespace Drupal\Tests\term_delete_protection\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;

/**
 * Tests the TermReferenceChecker service.
 *
 * @group term_delete_protection
 */
class TermReferenceCheckerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'node',
    'taxonomy',
    'field',
    'text',
    'term_delete_protection',
  ];

  /**
   * The term reference checker service.
   *
   * @var \Drupal\term_delete_protection\Service\TermReferenceChecker
   */
  protected $termReferenceChecker;

  /**
   * A test vocabulary.
   *
   * @var \Drupal\taxonomy\VocabularyInterface
   */
  protected $vocabulary;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installEntitySchema('taxonomy_term');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['node', 'taxonomy', 'term_delete_protection']);

    // Create a test vocabulary.
    $this->vocabulary = Vocabulary::create([
      'vid' => 'test_vocab',
      'name' => 'Test Vocabulary',
    ]);
    $this->vocabulary->save();

    // Create a content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ]);
    $node_type->save();

    // Create a taxonomy reference field on the article content type.
    FieldStorageConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Tags',
      'settings' => [
        'handler' => 'default:taxonomy_term',
        'handler_settings' => [
          'target_bundles' => [
            'test_vocab' => 'test_vocab',
          ],
        ],
      ],
    ])->save();

    $this->termReferenceChecker = $this->container->get('term_delete_protection.reference_checker');
  }

  /**
   * Tests that unreferenced terms can be identified.
   */
  public function testUnreferencedTerm() {
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Unreferenced Term',
    ]);
    $term->save();

    // Configure protection for nodes.
    $config = $this->config('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    $result = $this->termReferenceChecker->checkTermReferences($term);
    $this->assertFalse($result['has_references'], 'Unreferenced term should not have references.');
    $this->assertEmpty($result['reason'], 'Reason should be empty for unreferenced term.');
  }

  /**
   * Tests that referenced terms are detected.
   */
  public function testReferencedTerm() {
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Referenced Term',
    ]);
    $term->save();

    // Create a published node referencing the term.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
      'field_tags' => [
        ['target_id' => $term->id()],
      ],
    ]);
    $node->save();

    // Configure protection for nodes.
    $config = $this->config('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    $result = $this->termReferenceChecker->checkTermReferences($term);
    $this->assertTrue($result['has_references'], 'Referenced term should have references.');
    $this->assertNotEmpty($result['reason'], 'Reason should be provided for referenced term.');
  }

  /**
   * Tests that unpublished nodes don't prevent deletion.
   */
  public function testUnpublishedNodeDoesNotPreventDeletion() {
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Test Term',
    ]);
    $term->save();

    // Create an unpublished node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Unpublished Article',
      'status' => 0,
      'field_tags' => [
        ['target_id' => $term->id()],
      ],
    ]);
    $node->save();

    // Configure protection for nodes.
    $config = $this->config('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    $result = $this->termReferenceChecker->checkTermReferences($term);
    $this->assertFalse($result['has_references'], 'Unpublished node should not prevent deletion.');
  }

  /**
   * Tests recursive descendant term checking.
   */
  public function testDescendantTermProtection() {
    // Create parent term.
    $parent_term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Parent Term',
    ]);
    $parent_term->save();

    // Create child term.
    $child_term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Child Term',
      'parent' => [$parent_term->id()],
    ]);
    $child_term->save();

    // Reference the child term in a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
      'field_tags' => [
        ['target_id' => $child_term->id()],
      ],
    ]);
    $node->save();

    // Configure protection for nodes.
    $config = $this->config('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    // Parent term should be protected because child is referenced.
    $result = $this->termReferenceChecker->checkTermReferences($parent_term);
    $this->assertTrue($result['has_references'], 'Parent term should be protected when descendant is referenced.');
    $this->assertStringContainsString('descendant', $result['reason'], 'Reason should mention descendants.');
  }

  /**
   * Tests that protection respects vocabulary configuration.
   */
  public function testProtectionRespectConfiguration() {
    $term = Term::create([
      'vid' => 'test_vocab',
      'name' => 'Test Term',
    ]);
    $term->save();

    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'status' => 1,
      'field_tags' => [
        ['target_id' => $term->id()],
      ],
    ]);
    $node->save();

    // No protection configured.
    $result = $this->termReferenceChecker->checkTermReferences($term);
    $this->assertFalse($result['has_references'], 'Term should not be protected when no configuration exists.');

    // Configure protection for nodes.
    $config = $this->config('term_delete_protection.settings');
    $config->set('vocabularies', [
      [
        'vocabulary_id' => 'test_vocab',
        'protected_entity_types' => ['node'],
      ],
    ])->save();

    $result = $this->termReferenceChecker->checkTermReferences($term);
    $this->assertTrue($result['has_references'], 'Term should be protected when configuration exists.');
  }

}
