<?php

declare(strict_types=1);

namespace Drupal\Tests\test_helpers\Unit\Utils;

use Drupal\Tests\UnitTestCase;
use Drupal\test_helpers\TestHelpers;
use Drupal\test_helpers\utils\AssertSequence;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\CoversMethod;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests AssertSequence utility class.
 */
#[CoversClass(AssertSequence::class)]
#[Group('test_helpers')]
#[CoversMethod(AssertSequence::class, '__construct')]
class AssertSequenceTest extends UnitTestCase {

  /**
   * A static instance to test in multiple functions.
   *
   * @var \Drupal\test_helpers\utils\AssertSequence
   */
  protected static $assertSequence;

  /**
   * Tests the callEventSubscriber method.
   */
  public function testCallEventSubscriber() {
    // A positive scenario with correct sequence.
    self::$assertSequence = NULL;
    $sequence = [
      'value1',
      'value2',
      'value1',
      'value1',
      '333',
    ];
    $this->callAssertForList($sequence);
    self::$assertSequence->__destruct();

    // Wrong order of values.
    self::$assertSequence = NULL;
    $sequence = [
      'value1',
      'value1',
      'value2',
      'value1',
      '333',
    ];
    TestHelpers::assertException(function () use ($sequence) {
      $this->callAssertForList($sequence);
    });
    self::$assertSequence->__destruct();

    // Too many values.
    self::$assertSequence = NULL;
    $sequence = [
      'value1',
      'value2',
      'value1',
      'value1',
      '333',
      '333',
    ];
    TestHelpers::assertException(function () use ($sequence) {
      $this->callAssertForList($sequence);
    });
    self::$assertSequence->__destruct();

    // Not enough values.
    self::$assertSequence = NULL;
    $sequence = [
      'value1',
      'value2',
      'value1',
      'value1',
    ];
    $this->callAssertForList($sequence);
    $assertSequence = &self::$assertSequence;
    TestHelpers::assertException(function () use ($assertSequence) {
      $assertSequence->finalize();
    });
    // No exception should be thrown, because we finalized manually.
    $assertSequence->__destruct();
    TestHelpers::setPrivateProperty($assertSequence, 'isExceptionThrown', TRUE);
  }

  /**
   * Calls the assertion for a list.
   *
   * @param array $list
   *   A list of values.
   */
  private function callAssertForList(array $list): void {
    foreach ($list as $item) {
      $this->assertSequenceCallback($item);
    }
  }

  /**
   * Asserts a single value of a sequence.
   *
   * @param mixed $value
   *   A value to assert.
   */
  private function assertSequenceCallback($value): void {
    $expectedSequence = [
      'value1',
      'value2',
      'value1',
      'value1',
      333,
    ];
    if (empty(self::$assertSequence)) {
      self::$assertSequence = new AssertSequence($expectedSequence, 'myList1');
    }
    self::$assertSequence->assert($value);
  }

}
