<?php

declare(strict_types=1);

namespace Drupal\test_helpers_http_client_mock\Controller;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponseInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\test_helpers_http_client_mock\HttpClientFactoryMock;
use GuzzleHttp\Psr7\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response as SymfonyResponse;

/**
 * Controller for the HttpClientMock test helper pages.
 */
class HttpClientMockController extends ControllerBase {

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The HTTP client factory service.
   *
   * @var \Drupal\test_helpers_http_client_mock\HttpClientFactoryMock
   */
  protected HttpClientFactoryMock $httpClientFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);
    $instance->stateService = $container->get('state');
    $instance->requestStack = $container->get('request_stack');
    $instance->httpClientFactory = $container->get('http_client_factory');
    $instance->moduleHandler = $container->get('module_handler');
    return $instance;
  }

  /**
   * Gets the settings for the HttpClientFactoryMock settings.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function stubGetSettings(): JsonResponse {
    return new JsonResponse($this->httpClientFactory->stubGetConfiguration()->getRawData());
  }

  /**
   * Sets the settings for the HttpClientFactoryMock settings.
   *
   * Pass the values via GET parameters:
   * - mode: The mode to use: 'store' or 'mock'.
   * - name: The test name.
   * - module: The module name, used to get the module path for the directory/
   * - directory: The directory to store the responses. Absolute path, or
   *   relative to the module path, if the module parameter is set.
   * - context: A context string to use for generating the hash of the stored
   *   responses. Useful to mock different responses for the same URL and body.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function stubSetSettings(): JsonResponse {
    $request = $this->requestStack->getCurrentRequest();
    $settings = json_decode($request->query->get('settings'), TRUE);
    if (array_key_exists('mode', $settings)) {
      $this->httpClientFactory->stubSetConfig(HttpClientFactoryMock::SETTING_KEY_REQUEST_MOCK_MODE, $settings['mode']);
    }
    if (array_key_exists('name', $settings)) {
      $this->httpClientFactory->stubSetConfig(HttpClientFactoryMock::SETTING_KEY_TEST_NAME, $settings['name']);
    }
    if (array_key_exists('context', $settings)) {
      $this->httpClientFactory->stubSetConfig(HttpClientFactoryMock::SETTING_KEY_CONTEXT, $settings['context']);
    }
    if (array_key_exists('directory', $settings)) {
      if (
        !str_starts_with($settings['directory'], '/')
        && isset($settings['module'])
      ) {
        $modulePath = $this->moduleHandler->getModule($settings['module'])->getPath();
        $directory = $modulePath . DIRECTORY_SEPARATOR . $settings['directory'];
      }
      else {
        $directory = $settings['directory'];
      }
      $this->httpClientFactory->stubSetConfig(HttpClientFactoryMock::SETTING_KEY_RESPONSES_STORAGE_DIRECTORY, $directory);
    }
    if (array_key_exists('uri_regexp', $settings)) {
      $this->httpClientFactory->stubSetConfig(HttpClientFactoryMock::SETTING_KEY_URI_REGEXP, $settings['uri_regexp']);
    }
    return new JsonResponse([
      'status' => 'success',
      'settings' => $this->httpClientFactory->stubGetConfiguration()->getRawData(),
    ]);
  }

  /**
   * Gets a stored response.
   *
   * @param mixed $hash
   *   The hash value.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function stubGetStoredResponse($hash): SymfonyResponse {
    if ($this->requestStack->getCurrentRequest()->query->get('metadata') == TRUE) {
      $body = $this->httpClientFactory->stubGetStoredResponseMetadataByHash($hash);
      $symfonyResponse = new SymfonyResponse(json_encode($body));
      $symfonyResponse->headers->set('Content-Type', 'application/json');
    }
    else {
      $response = $this->httpClientFactory->stubGetStoredResponseByHash($hash);
      $body = $response->getBody()->getContents();
      $symfonyResponse = new SymfonyResponse($body, $response->getStatusCode(), $response->getHeaders());
      $symfonyResponse->headers->set('Content-Type', 'text/plain');
    }

    // Add headers to make the HTTP response non-cacheable by browsers and
    // proxies.
    $symfonyResponse->headers->set('Cache-Control', 'no-store');

    // Ensure the response is not cacheable by Drupal's internal cache systems.
    if ($symfonyResponse instanceof CacheableResponseInterface) {
      $symfonyResponse->addCacheableDependency((new CacheableMetadata())->setCacheMaxAge(0));
    }
    return $symfonyResponse;
  }

  /**
   * Delete the stored response files.
   *
   * @param mixed $hash
   *   The hash value.
   *
   * @return \GuzzleHttp\Psr7\Response
   *   The response.
   */
  public function stubDeleteStoredResponse($hash): JsonResponse {
    $this->httpClientFactory->stubDeleteStoredResponseByHash($hash);
    return new JsonResponse(['status' => 'success']);
  }

  /**
   * Sets the stored response data.
   *
   * @param mixed $hash
   *   The hash value.
   *
   * @return \GuzzleHttp\Psr7\Response
   *   The response.
   */
  public function stubSetStoredResponse($hash): JsonResponse {
    $request = $this->requestStack->getCurrentRequest();
    $data = $request->getContent();
    $status = $request->query->get('status', 200);
    $headers = $request->query->has('headers')
      ? json_decode($request->query->get('headers'))
      : [];
    $response = new Response($status, $headers, $data);
    $this->httpClientFactory->stubStoreResponse($response, NULL, $hash);
    return new JsonResponse(['status' => 'success']);
  }

  /**
   * Gets the last requests hashes.
   *
   * @return \GuzzleHttp\Psr7\Response
   *   The response.
   */
  public function stubGetLastRequestsHashes(): JsonResponse {
    $data = $this->stateService->get(HttpClientFactoryMock::STATE_KEY_LAST_REQUESTS_HASHES, []);
    return new JsonResponse($data);
  }

  /**
   * Gets the last response.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function stubGetLastResponse($delta = 0): SymfonyResponse {
    $lastHashes = $this->stateService->get(HttpClientFactoryMock::STATE_KEY_LAST_REQUESTS_HASHES, []);
    $hash = $lastHashes[$delta] ?? NULL;
    return $this->stubGetStoredResponse($hash);
  }

}
