<?php

declare(strict_types=1);

namespace Drupal\Tests\tester\Unit\Plugin\Tester;

use Drupal\Core\Menu\MenuLinkInterface;
use Drupal\Core\Menu\MenuLinkTreeElement;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\Url;
use Drupal\tester\Plugin\Tester\MenuTester;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tests the MenuTester plugin.
 *
 * @group tester
 * @coversDefaultClass \Drupal\tester\Plugin\Tester\MenuTester
 */
class MenuTesterTest extends UnitTestCase {

  /**
   * The menu tester plugin.
   *
   * @var \Drupal\tester\Plugin\Tester\MenuTester
   */
  protected $menuTester;

  /**
   * Mock menu link tree service.
   *
   * @var \Drupal\Core\Menu\MenuLinkTreeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $menuTree;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->menuTree = $this->createMock(MenuLinkTreeInterface::class);

    $configuration = [];
    $plugin_id = 'menu';
    $plugin_definition = [];

    $this->menuTester = new MenuTester(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $this->menuTree
    );
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = $this->createMock(ContainerInterface::class);
    $container
      ->method('get')
      ->with('menu.link_tree')
      ->willReturn($this->menuTree);

    $instance = MenuTester::create(
      $container,
      [],
      'menu',
      []
    );

    $this->assertInstanceOf(MenuTester::class, $instance);
  }

  /**
   * Tests urls method with single menu.
   *
   * @covers ::urls
   * @covers ::buildUrls
   */
  public function testUrlsWithSingleMenu(): void {
    $options = ['limit' => 0, 'menus' => 'main'];

    // Create mock menu link.
    $url = $this->createMock(Url::class);
    $url->method('isExternal')->willReturn(FALSE);
    $url->method('toString')->willReturn('/test-link');

    $menuLink = $this->createMock(MenuLinkInterface::class);
    $menuLink->method('getUrlObject')->willReturn($url);
    $menuLink->method('isEnabled')->willReturn(TRUE);

    // Create mock menu tree element.
    $treeElement = $this->createMock(MenuLinkTreeElement::class);
    $treeElement->link = $menuLink;
    $treeElement->subtree = [];

    $tree = ['test-link' => $treeElement];

    $this->menuTree
      ->expects($this->once())
      ->method('load')
      ->with('main', $this->isInstanceOf(MenuTreeParameters::class))
      ->willReturn($tree);

    $this->menuTree
      ->expects($this->once())
      ->method('transform')
      ->with($tree, [['callable' => 'menu.default_tree_manipulators:generateIndexAndSort']])
      ->willReturn($tree);

    $urls = $this->menuTester->urls($options);

    $expected = ['/test-link'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests urls method with multiple menus.
   *
   * @covers ::urls
   * @covers ::buildUrls
   */
  public function testUrlsWithMultipleMenus(): void {
    $options = ['limit' => 0, 'menus' => 'main,footer'];

    // Create mock URLs.
    $url1 = $this->createMock(Url::class);
    $url1->method('isExternal')->willReturn(FALSE);
    $url1->method('toString')->willReturn('/main-link');

    $url2 = $this->createMock(Url::class);
    $url2->method('isExternal')->willReturn(FALSE);
    $url2->method('toString')->willReturn('/footer-link');

    // Create mock menu links.
    $menuLink1 = $this->createMock(MenuLinkInterface::class);
    $menuLink1->method('getUrlObject')->willReturn($url1);
    $menuLink1->method('isEnabled')->willReturn(TRUE);

    $menuLink2 = $this->createMock(MenuLinkInterface::class);
    $menuLink2->method('getUrlObject')->willReturn($url2);
    $menuLink2->method('isEnabled')->willReturn(TRUE);

    // Create mock tree elements.
    $treeElement1 = $this->createMock(MenuLinkTreeElement::class);
    $treeElement1->link = $menuLink1;
    $treeElement1->subtree = [];

    $treeElement2 = $this->createMock(MenuLinkTreeElement::class);
    $treeElement2->link = $menuLink2;
    $treeElement2->subtree = [];

    $mainTree = ['main-link' => $treeElement1];
    $footerTree = ['footer-link' => $treeElement2];

    $this->menuTree
      ->expects($this->exactly(2))
      ->method('load')
      ->willReturnCallback(function ($menu_name) use ($mainTree, $footerTree) {
        return $menu_name === 'main' ? $mainTree : $footerTree;
      });

    $this->menuTree
      ->expects($this->exactly(2))
      ->method('transform')
      ->willReturnCallback(function ($tree) {
        return $tree;
      });

    $urls = $this->menuTester->urls($options);

    $expected = ['/main-link', '/footer-link'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests urls method filters external links.
   *
   * @covers ::urls
   * @covers ::buildUrls
   */
  public function testUrlsFiltersExternalLinks(): void {
    $options = ['limit' => 0, 'menus' => 'main'];

    // Create external URL.
    $externalUrl = $this->createMock(Url::class);
    $externalUrl->method('isExternal')->willReturn(TRUE);

    // Create internal URL.
    $internalUrl = $this->createMock(Url::class);
    $internalUrl->method('isExternal')->willReturn(FALSE);
    $internalUrl->method('toString')->willReturn('/internal-link');

    // Create mock menu links.
    $externalMenuLink = $this->createMock(MenuLinkInterface::class);
    $externalMenuLink->method('getUrlObject')->willReturn($externalUrl);
    $externalMenuLink->method('isEnabled')->willReturn(TRUE);

    $internalMenuLink = $this->createMock(MenuLinkInterface::class);
    $internalMenuLink->method('getUrlObject')->willReturn($internalUrl);
    $internalMenuLink->method('isEnabled')->willReturn(TRUE);

    // Create mock tree elements.
    $externalElement = $this->createMock(MenuLinkTreeElement::class);
    $externalElement->link = $externalMenuLink;
    $externalElement->subtree = [];

    $internalElement = $this->createMock(MenuLinkTreeElement::class);
    $internalElement->link = $internalMenuLink;
    $internalElement->subtree = [];

    $tree = [
      'external' => $externalElement,
      'internal' => $internalElement,
    ];

    $this->menuTree
      ->method('load')
      ->willReturn($tree);

    $this->menuTree
      ->method('transform')
      ->willReturn($tree);

    $urls = $this->menuTester->urls($options);

    // Should only contain internal link.
    $expected = ['/internal-link'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests urls method with limit.
   *
   * @covers ::urls
   */
  public function testUrlsWithLimit(): void {
    $options = ['limit' => 1, 'menus' => 'main'];

    // Create multiple internal URLs.
    $url1 = $this->createMock(Url::class);
    $url1->method('isExternal')->willReturn(FALSE);
    $url1->method('toString')->willReturn('/link-1');

    $url2 = $this->createMock(Url::class);
    $url2->method('isExternal')->willReturn(FALSE);
    $url2->method('toString')->willReturn('/link-2');

    $menuLink1 = $this->createMock(MenuLinkInterface::class);
    $menuLink1->method('getUrlObject')->willReturn($url1);
    $menuLink1->method('isEnabled')->willReturn(TRUE);

    $menuLink2 = $this->createMock(MenuLinkInterface::class);
    $menuLink2->method('getUrlObject')->willReturn($url2);
    $menuLink2->method('isEnabled')->willReturn(TRUE);

    $treeElement1 = $this->createMock(MenuLinkTreeElement::class);
    $treeElement1->link = $menuLink1;
    $treeElement1->subtree = [];

    $treeElement2 = $this->createMock(MenuLinkTreeElement::class);
    $treeElement2->link = $menuLink2;
    $treeElement2->subtree = [];

    $tree = [
      'link-1' => $treeElement1,
      'link-2' => $treeElement2,
    ];

    $this->menuTree
      ->method('load')
      ->willReturn($tree);

    $this->menuTree
      ->method('transform')
      ->willReturn($tree);

    $urls = $this->menuTester->urls($options);

    // Should be limited to 1 URL.
    $expected = ['/link-1'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests dependencies method.
   *
   * @covers ::dependencies
   */
  public function testDependencies(): void {
    $dependencies = $this->menuTester->dependencies();

    $expected = [
      'modules' => ['system'],
    ];

    $this->assertEquals($expected, $dependencies);
  }

}
