<?php

declare(strict_types=1);

namespace Drupal\Tests\tester\Unit\Plugin\Tester;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\tester\Plugin\Tester\UserTester;
use Drupal\Tests\UnitTestCase;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tests the UserTester plugin.
 *
 * @group tester
 * @coversDefaultClass \Drupal\tester\Plugin\Tester\UserTester
 */
class UserTesterTest extends UnitTestCase {

  /**
   * The user tester plugin.
   *
   * @var \Drupal\tester\Plugin\Tester\UserTester
   */
  protected $userTester;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock user storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $userStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->userStorage = $this->createMock(EntityStorageInterface::class);

    $this->entityTypeManager
      ->method('getStorage')
      ->with('user')
      ->willReturn($this->userStorage);

    $configuration = [];
    $plugin_id = 'user';
    $plugin_definition = [];

    $this->userTester = new UserTester(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $this->entityTypeManager
    );
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = $this->createMock(ContainerInterface::class);
    $container
      ->method('get')
      ->with('entity_type.manager')
      ->willReturn($this->entityTypeManager);

    $instance = UserTester::create(
      $container,
      [],
      'user',
      []
    );

    $this->assertInstanceOf(UserTester::class, $instance);
  }

  /**
   * Tests urls method with no limit.
   *
   * @covers ::urls
   */
  public function testUrlsWithNoLimit(): void {
    $options = ['limit' => 0];

    // Mock users.
    $user1 = $this->createMock(UserInterface::class);
    $user1->method('id')->willReturn('1');
    $user1->method('toUrl')->willReturn($this->getMockUrl('/user/1'));

    $user2 = $this->createMock(UserInterface::class);
    $user2->method('id')->willReturn('2');
    $user2->method('toUrl')->willReturn($this->getMockUrl('/user/2'));

    $users = [1 => $user1, 2 => $user2];

    $this->userStorage
      ->method('loadMultiple')
      ->willReturn($users);

    $urls = $this->userTester->urls($options);

    $expected = ['/user/1', '/user/2'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests urls method with limit.
   *
   * @covers ::urls
   */
  public function testUrlsWithLimit(): void {
    $options = ['limit' => 1];

    // Mock users.
    $user1 = $this->createMock(UserInterface::class);
    $user1->method('id')->willReturn('1');
    $user1->method('toUrl')->willReturn($this->getMockUrl('/user/1'));

    $user2 = $this->createMock(UserInterface::class);
    $user2->method('id')->willReturn('2');
    $user2->method('toUrl')->willReturn($this->getMockUrl('/user/2'));

    $users = [1 => $user1, 2 => $user2];

    $this->userStorage
      ->method('loadMultiple')
      ->willReturn($users);

    $urls = $this->userTester->urls($options);

    // Should be limited to 1 URL.
    $expected = ['/user/1'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests dependencies method.
   *
   * @covers ::dependencies
   */
  public function testDependencies(): void {
    $dependencies = $this->userTester->dependencies();

    $expected = [
      'modules' => ['user'],
    ];

    $this->assertEquals($expected, $dependencies);
  }

  /**
   * Helper to create a mock URL.
   *
   * @param string $path
   *   The URL path.
   *
   * @return \Drupal\Core\Url|\PHPUnit\Framework\MockObject\MockObject
   *   Mock URL object.
   */
  protected function getMockUrl(string $path) {
    $url = $this->createMock('\Drupal\Core\Url');
    $url->method('toString')->willReturn($path);
    return $url;
  }

}
