<?php

declare(strict_types=1);

namespace Drupal\Tests\tester\Unit\Plugin\Tester;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeInterface;
use Drupal\node\NodeTypeInterface;
use Drupal\tester\Plugin\Tester\NodeTester;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Tests the NodeTester plugin.
 *
 * @group tester
 * @coversDefaultClass \Drupal\tester\Plugin\Tester\NodeTester
 */
class NodeTesterTest extends UnitTestCase {

  /**
   * The node tester plugin.
   *
   * @var \Drupal\tester\Plugin\Tester\NodeTester
   */
  protected $nodeTester;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock node storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $nodeStorage;

  /**
   * Mock node type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $nodeTypeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->nodeStorage = $this->createMock(EntityStorageInterface::class);
    $this->nodeTypeStorage = $this->createMock(EntityStorageInterface::class);

    $this->entityTypeManager
      ->method('getStorage')
      ->willReturnMap([
        ['node_type', $this->nodeTypeStorage],
        ['node', $this->nodeStorage],
      ]);

    $configuration = [];
    $plugin_id = 'node';
    $plugin_definition = [];

    $this->nodeTester = new NodeTester(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $this->entityTypeManager
    );
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = $this->createMock(ContainerInterface::class);
    $container
      ->method('get')
      ->with('entity_type.manager')
      ->willReturn($this->entityTypeManager);

    $instance = NodeTester::create(
      $container,
      [],
      'node',
      []
    );

    $this->assertInstanceOf(NodeTester::class, $instance);
  }

  /**
   * Tests urls method with no limit.
   *
   * @covers ::urls
   */
  public function testUrlsWithNoLimit(): void {
    $options = ['limit' => 0];

    // Mock node types.
    $nodeType1 = $this->createMock(NodeTypeInterface::class);
    $nodeType1->method('id')->willReturn('article');

    $nodeType2 = $this->createMock(NodeTypeInterface::class);
    $nodeType2->method('id')->willReturn('page');

    $nodeTypes = ['article' => $nodeType1, 'page' => $nodeType2];

    $this->nodeTypeStorage
      ->method('loadMultiple')
      ->willReturn($nodeTypes);

    // Mock nodes.
    $node1 = $this->createMock(NodeInterface::class);
    $node1->method('toUrl')->willReturn($this->getMockUrl('/node/1'));

    $node2 = $this->createMock(NodeInterface::class);
    $node2->method('toUrl')->willReturn($this->getMockUrl('/node/2'));

    $nodes = [1 => $node1, 2 => $node2];

    $this->nodeStorage
      ->method('loadMultiple')
      ->willReturn($nodes);

    $urls = $this->nodeTester->urls($options);

    $expected = ['/node/add/article', '/node/add/page', '/node/1', '/node/2'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests urls method with limit.
   *
   * @covers ::urls
   */
  public function testUrlsWithLimit(): void {
    $options = ['limit' => 2];

    // Mock node types.
    $nodeType1 = $this->createMock(NodeTypeInterface::class);
    $nodeType1->method('id')->willReturn('article');

    $nodeType2 = $this->createMock(NodeTypeInterface::class);
    $nodeType2->method('id')->willReturn('page');

    $nodeTypes = ['article' => $nodeType1, 'page' => $nodeType2];

    $this->nodeTypeStorage
      ->method('loadMultiple')
      ->willReturn($nodeTypes);

    // Mock nodes.
    $node1 = $this->createMock(NodeInterface::class);
    $node1->method('toUrl')->willReturn($this->getMockUrl('/node/1'));

    $node2 = $this->createMock(NodeInterface::class);
    $node2->method('toUrl')->willReturn($this->getMockUrl('/node/2'));

    $nodes = [1 => $node1, 2 => $node2];

    $this->nodeStorage
      ->method('loadMultiple')
      ->willReturn($nodes);

    $urls = $this->nodeTester->urls($options);

    // Should be limited to 2 URLs.
    $expected = ['/node/add/article', '/node/add/page'];
    $this->assertEquals($expected, $urls);
  }

  /**
   * Tests dependencies method.
   *
   * @covers ::dependencies
   */
  public function testDependencies(): void {
    $dependencies = $this->nodeTester->dependencies();

    $expected = [
      'modules' => ['node', 'user'],
    ];

    $this->assertEquals($expected, $dependencies);
  }

  /**
   * Helper to create a mock URL.
   *
   * @param string $path
   *   The URL path.
   *
   * @return \Drupal\Core\Url|\PHPUnit\Framework\MockObject\MockObject
   *   Mock URL object.
   */
  protected function getMockUrl(string $path) {
    $url = $this->createMock('\Drupal\Core\Url');
    $url->method('toString')->willReturn($path);
    return $url;
  }

}
