<?php

namespace Drupal\tether_stats\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\tether_stats\TetherStatsManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form to purge activity data before a given date.
 */
class TetherStatsActivityPurgeConfirmForm extends ConfirmFormBase {

  /**
   * The manager service.
   *
   * @var \Drupal\tether_stats\TetherStatsManagerInterface
   */
  private $manager;

  /**
   * The date before which to purge all activity data from the log.
   *
   * @var string
   */
  protected string $purgeBeforeDate;

  /**
   * Constructor.
   *
   * @param \Drupal\tether_stats\TetherStatsManagerInterface $manager
   *   The manager service.
   */
  public function __construct(TetherStatsManagerInterface $manager) {

    $this->manager = $manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('tether_stats.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $purge_before_date = NULL): array {

    $this->purgeBeforeDate = $purge_before_date;

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {

    $current_time = new \DateTime();
    $current_time->setTimestamp(\Drupal::time()->getRequestTime());

    $purge_before_time = strtotime($this->purgeBeforeDate);

    if (!$purge_before_time) {

      $form_state->setError($form, $this->t('Invalid date specified in route.'));
    }
    elseif ($purge_before_time >= $current_time->getTimestamp()) {

      $form_state->setError($form, $this->t('The purge before date must be earlier than the current date.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    $database = $this->manager->getDatabaseConnection();
    $purge_before_time = strtotime($this->purgeBeforeDate);

    $subquery = $database->select('tether_stats_activity_log', 'a')
      ->fields('a', ['alid'])
      ->condition('created', $purge_before_time, '<');

    $database->delete('tether_stats_impression_log')
      ->condition('alid', $subquery, 'IN')
      ->execute();

    $database->delete('tether_stats_activity_log')
      ->condition('created', $purge_before_time, '<')
      ->execute();

    $this->messenger()->addStatus($this->t('All activity and impression data before %date has been purged from the database.',
      [
        '%date' => $this->purgeBeforeDate,
      ]));

    $form_state->setRedirect('tether_stats.activity_purge_form');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() : string {
    return "tether_stats_activity_purge_confirm_form";
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('tether_stats.activity_purge_form');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): string {

    return $this->t('Are you sure you want to purge all activity in the log before %date?', ['%date' => $this->purgeBeforeDate]);
  }

}
