<?php

namespace Drupal\tether_stats\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Form to purge activity data before a given date.
 */
class TetherStatsActivityPurgeForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form = [];

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>This form will <strong>permanently</strong> delete old activity and impression data from the database.</p><p>Please make sure you have backed up your data before proceeding and have set the date carefully.</p>',
    ];

    $form['purge_before'] = [
      '#type' => 'date',
      '#title' => $this->t('Purge before date'),
      '#description' => $this->t('Purge all data from the Activity Log before the specified date.'),
      '#required' => TRUE,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Purge data from Activity Log'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {

    $current_time = new \DateTime();
    $current_time->setTimestamp(\Drupal::time()->getRequestTime());

    $purge_before_time = strtotime($form_state->getValue('purge_before'));

    if (!$purge_before_time) {

      $form_state->setError($form['purge_before'], $this->t('Invalid date specified in route.'));
    }
    elseif ($purge_before_time >= $current_time->getTimestamp()) {

      $form_state->setError($form['purge_before'], $this->t('The purge before date must be before the current date.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    $form_state->setRedirect('tether_stats.activity_purge_confirm_form', [
      'purge_before_date' => $form_state->getValue('purge_before')]);
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() : string {
    return "tether_stats_activity_purge_form";
  }

}
