(function ($, Drupal, once) {
  // Drupal.t not work with once : https://www.drupal.org/project/drupal/issues/3413709
  const errorNotFoundClass = Drupal.t(
    'The block is not configured for this type of content.',
  );
  const errorOccurred = Drupal.t(
    'An error has occurred when calculating ux/ui.',
  );

  /**
   * Display an error message and hide other elements
   *
   * @param {string} errorMessage
   *  The formatted error by Drupal.t
   *
   * @returns {void}
   */
  function setAnError(errorMessage) {
    const error = document.querySelector('.metric-error');
    error.innerHTML = errorMessage;

    const metricsContainers = document.querySelectorAll(
      [
        '.metric-text',
        '.metric-reading',
        '.metric-paragraph',
        '.metric-paragraph-len',
        '.metric-img',
        '.metric-decorative-img',
        '.metric-background-img',
        '.metric-valid-title',
        '.metric-heading-depth',
        '.metric-internal-links',
        '.metric-external-links',
        '.metric-element',
        '.metric-load-time',
        '.metric-score',
        '.metric-hr',
      ].join(','),
    );

    metricsContainers.forEach((container) => {
      container.style.display = 'none';
    });
  }

  /**
   * Calculate the ux score
   *
   * @param {Array} metrics
   *  The metrics calculated
   *
   * @param {Array} config
   *  The module configurations
   *
   * @returns {number}
   *  The ux score
   */
  function calculateUXScore(metrics, config) {
    let score = 100;
    let enabledCategories = 0;

    if (config.text_metrics) {
      enabledCategories++;
      if (metrics.textLength > 1000) {
        score -= 10;
      }
      if (metrics.paragraphCount < 3) {
        score -= 10;
      }
      if (metrics.averageParagraphLength > 80) {
        score -= 10;
      }
    }

    if (config.images) {
      enabledCategories++;
      if (metrics.imageCount < 1) {
        score -= 15;
      }
    }

    if (config.titles) {
      enabledCategories++;
      if (!metrics.hasValidHeadingStructure) {
        score -= 20;
      }
      if (metrics.headingDepth > 4) {
        score -= 5;
      }
    }

    if (config.links) {
      enabledCategories++;
      if (metrics.externalLinkCount > metrics.internalLinkCount) {
        score -= 5;
      }
    }

    if (config.performance) {
      enabledCategories++;
      if (metrics.elementCount > 50) {
        score -= 10;
      }
    }

    if (enabledCategories > 0) {
      score = (score / enabledCategories) * enabledCategories;
    }

    return Math.max(0, score);
  }

  /**
   * Check the title structure on node
   *
   * @param {Array} headings
   *  The node titles
   *
   * @returns {boolean}
   *  true if structure is good
   */
  function checkHeadingStructure(headings) {
    if (headings.length === 0) {
      return true;
    }

    let lastLevel = 0;
    let isValid = true;
    headings.forEach((heading) => {
      const level = parseInt(heading.tagName.replace('H', ''), 10);
      if (lastLevel && level > lastLevel + 1) {
        isValid = false;
      }
      lastLevel = level;
    });
    return isValid;
  }

  /**
   * Update metrics in modal based on the config
   *
   * @param {Array} metrics
   *  The metrics calculated
   *
   * @param {number} score
   *  The ux score
   *
   * @param {Array} config
   *  The module configurations & recommendations
   *
   * @returns {void}
   */
  function updateMetrics(metrics, score, config) {
    try {
      const elements = {};

      elements['.metric-reading'] = `${metrics.readingTime} min`;

      if (config.text_metrics) {
        elements['.metric-text'] = metrics.textLength;
        elements['.metric-paragraph'] = metrics.paragraphCount;
        elements['.metric-paragraph-len'] =
          metrics.averageParagraphLength.toFixed(2);
      }

      if (config.images) {
        elements['.metric-img'] = metrics.imageCount;
        elements['.metric-decorative-img'] = metrics.decorativeImageCount;
        elements['.metric-background-img'] = metrics.backgroundImageCount;
      }

      if (config.titles) {
        elements['.metric-valid-title'] = metrics.hasValidHeadingStructure
          ? '✅'
          : '❌';
        elements['.metric-heading-depth'] = metrics.headingDepth;
      }

      if (config.links) {
        elements['.metric-internal-links'] = metrics.internalLinkCount;
        elements['.metric-external-links'] = metrics.externalLinkCount;
      }

      if (config.performance) {
        elements['.metric-element'] = metrics.elementCount;
        elements['.metric-load-time'] = `${metrics.estimatedLoadTime.toFixed(
          2,
        )} sec`;
      }

      elements['#score'] = score;

      Object.entries(elements).forEach(([selector, value]) => {
        const element = document.querySelector(selector);
        if (element) {
          element.textContent = `${element.textContent.split(':')[0]}: ${value}`;
        }
      });
    } catch (error) {
      setAnError(errorOccurred);
    }
  }

  /**
   * Add a color to the button based on the score
   *
   * @param {number} score
   *  The ux score
   *
   * @returns {void}
   */
  function updateScoreCircle(score) {
    try {
      const scoreCircle = document.querySelector('.score-circle');
      if (!scoreCircle) {
        return;
      }

      const startColor = { r: 193, g: 26, b: 14 };
      const endColor = { r: 6, g: 169, b: 44 };

      const r = Math.floor(
        startColor.r + (endColor.r - startColor.r) * (score / 100),
      );
      const g = Math.floor(
        startColor.g + (endColor.g - startColor.g) * (score / 100),
      );
      const b = Math.floor(
        startColor.b + (endColor.b - startColor.b) * (score / 100),
      );

      scoreCircle.style.backgroundColor = `rgb(${r}, ${g}, ${b})`;
    } catch (error) {
      setAnError(errorOccurred);
    }
  }

  /**
   * Add the ux score in the button
   *
   * @param {number} score
   *  The ux/ui score
   *
   * @returns {void}
   */
  function setupScoreToggle(score) {
    try {
      const scorePastille = document.querySelector('#score-pastille');
      if (!scorePastille) {
        return;
      }

      scorePastille.textContent = score;
    } catch (error) {
      setAnError(errorOccurred);
    }
  }

  /**
   * Set up the modal to open or close when the button is clicked
   *
   * @param {HTMLElement} modal
   *  The text clarity checker modal
   *
   * @returns {void}
   */
  function setupModalToggle(modal) {
    try {
      const scorePastille = document.querySelector('#score-pastille');
      if (!scorePastille) {
        return;
      }

      scorePastille.addEventListener('click', () => {
        modal.classList.toggle('show');
      });
    } catch (error) {
      setAnError(errorOccurred);
    }
  }

  Drupal.behaviors.textClarityChecker = {
    attach(context, settings) {
      const config = settings.textClarityChecker || {};

      once('textClarityChecker', '#text-clarity-checker', context).forEach(
        (modal) => {
          const classArray = ['node'];
          if (config.classes && config.classes !== '') {
            classArray.push(
              ...(config.classes.includes(';')
                ? config.classes.split(';')
                : [config.classes]),
            );
          }

          let node;
          classArray.forEach((className) => {
            if (document.querySelector(`.${className}`)) {
              node = document.querySelector(`.${className}`);
            }
          });

          if (!modal) {
            return;
          }

          setupModalToggle(modal);

          if (node === undefined || node === null) {
            setAnError(errorNotFoundClass);
            return;
          }

          const metrics = {
            textLength: 0,
            imageCount: 0,
            decorativeImageCount: 0,
            paragraphCount: 0,
            averageParagraphLength: 0,
            readingTime: 0,
            hasH1: false,
            hasValidHeadingStructure: true,
            backgroundImageCount: 0,
            elementCount: 0,
            internalLinkCount: 0,
            externalLinkCount: 0,
            headingDepth: 0,
            estimatedLoadTime: 0,
          };

          try {
            const textContent = node.textContent.trim();
            metrics.textLength = textContent.split(/\s+/).length;
            metrics.readingTime = Math.ceil(metrics.textLength / 200);

            if (config.text_metrics) {
              metrics.paragraphCount = node.getElementsByTagName('p').length;
              const paragraphs = Array.from(node.getElementsByTagName('p'));
              if (paragraphs.length > 0) {
                const totalParagraphLength = paragraphs.reduce(
                  (sum, p) => sum + p.textContent.trim().split(/\s+/).length,
                  0,
                );
                metrics.averageParagraphLength =
                  totalParagraphLength / paragraphs.length;
              }
            }

            if (config.images) {
              const images = Array.from(node.getElementsByTagName('img'));
              metrics.imageCount = images.length;

              images.forEach((img) => {
                const altText = img.alt.trim();
                if (!altText || altText.split(/\s+/).length < 5) {
                  metrics.decorativeImageCount++;
                  metrics.readingTime += 2;
                } else {
                  metrics.readingTime += Math.ceil(
                    altText.split(/\s+/).length / 200,
                  );
                }
              });

              const allElements = node.querySelectorAll('*');
              allElements.forEach((element) => {
                const style = window.getComputedStyle(element);
                if (style.backgroundImage !== 'none') {
                  metrics.backgroundImageCount++;
                }
              });
            }

            if (config.titles) {
              metrics.hasH1 = !!node.querySelector('h1');
              const headings = Array.from(
                node.querySelectorAll('h1, h2, h3, h4, h5, h6'),
              );
              metrics.hasValidHeadingStructure =
                checkHeadingStructure(headings);
              if (headings.length > 0) {
                metrics.headingDepth = Math.max(
                  ...headings.map((h) =>
                    parseInt(h.tagName.replace('H', ''), 10),
                  ),
                );
              }
            }

            if (config.links) {
              const links = Array.from(node.getElementsByTagName('a'));
              links.forEach((link) => {
                const href = link.getAttribute('href');
                if (href && href.startsWith(window.location.origin)) {
                  metrics.internalLinkCount++;
                } else if (href && href.startsWith('http')) {
                  metrics.externalLinkCount++;
                }
              });
            }

            if (config.performance) {
              metrics.elementCount = node.querySelectorAll('*').length;
              metrics.estimatedLoadTime =
                metrics.imageCount * 0.5 + metrics.elementCount * 0.01;
            }
            const score = calculateUXScore(metrics, config);
            updateMetrics(metrics, score, config);
            updateScoreCircle(score);
            setupScoreToggle(score);
          } catch (error) {
            setAnError(errorOccurred);
          }
        },
      );
    },
  };
})(jQuery, Drupal, once);
