<?php

namespace Drupal\text_clarity_checker\Plugin\Block;

use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides a text clarity checker Block.
 */
#[Block(
  id: 'text_clarity_checker_block',
  admin_label: new TranslatableMarkup("Text clarity checker block"),
  category: new TranslatableMarkup("Blocks"),
  context_definitions: [
    "node" => new EntityContextDefinition(
      'entity:node',
      label: new TranslatableMarkup("Current Node")
    ),
  ]
)]
class TextClarityCheckerBlock extends BlockBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'text_metrics' => TRUE,
      'images' => TRUE,
      'recommended_image_metrics' => 1,
      'links' => TRUE,
      'recommended_link_metrics' => 1,
      'recommended_external_link_metrics' => 1,
      'titles' => TRUE,
      'recommended_title_metrics' => 4,
      'performance' => TRUE,
      'classes' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['metrics_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Metrics Settings'),
      '#open' => TRUE,
    ];

    $form['metrics_settings']['text_metrics'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable text metrics analysis'),
      '#description' => $this->t('Check text length, reading time and paragraphs metrics.'),
      '#default_value' => $config['text_metrics'],
    ];

    $form['metrics_settings']['images'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable images analysis'),
      '#description' => $this->t('Check images count, decorative images and background images.'),
      '#default_value' => $config['images'],
    ];

    $form['metrics_settings']['recommended_image_metrics'] = [
      '#type' => 'number',
      '#title' => $this->t('Recommended number of images per content'),
      '#description' => $this->t('Set the optimal number of content images.'),
      '#min' => 0,
      '#max' => 20,
      '#step' => 1,
      '#default_value' => $config['recommended_image_metrics'],
    ];

    $form['metrics_settings']['links'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable links analysis'),
      '#description' => $this->t('Check internal and external links count.'),
      '#default_value' => $config['links'],
    ];

    $form['metrics_settings']['recommended_link_metrics'] = [
      '#type' => 'number',
      '#title' => $this->t('Recommended number of links per content'),
      '#description' => $this->t('Set the optimal number of links internal or external.'),
      '#min' => 0,
      '#max' => 20,
      '#step' => 1,
      '#default_value' => $config['recommended_link_metrics'],
    ];

    $form['metrics_settings']['recommended_external_link_metrics'] = [
      '#type' => 'number',
      '#title' => $this->t('Recommended number of external links per content'),
      '#description' => $this->t('Set the optimal number of external links.'),
      '#min' => 0,
      '#max' => 20,
      '#step' => 1,
      '#default_value' => $config['recommended_external_link_metrics'],
    ];

    $form['metrics_settings']['titles'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable titles structure analysis'),
      '#description' => $this->t('Check headings hierarchy and structure.'),
      '#default_value' => $config['titles'],
    ];

    $form['metrics_settings']['recommended_title_metrics'] = [
      '#type' => 'number',
      '#title' => $this->t('Recommended number of title step'),
      '#description' => $this->t('Set the maximal title step of a content.'),
      '#min' => 0,
      '#max' => 6,
      '#step' => 1,
      '#default_value' => $config['recommended_title_metrics'],
    ];

    $form['metrics_settings']['performance'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable performance analysis'),
      '#description' => $this->t('Check DOM elements count and estimated load time.'),
      '#default_value' => $config['performance'],
    ];

    $form['metrics_settings']['classes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Add classes to check'),
      '#description' => $this->t("You can add several classes separated by ;. Don't put a . in front of the class."),
      '#default_value' => $config['classes'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state): void {
    parent::blockSubmit($form, $form_state);
    $values = $form_state->getValues();
    $this->configuration['text_metrics'] = $values['metrics_settings']['text_metrics'];
    $this->configuration['images'] = $values['metrics_settings']['images'];
    $this->configuration['recommended_image_metrics'] = $values['metrics_settings']['recommended_image_metrics'];
    $this->configuration['links'] = $values['metrics_settings']['links'];
    $this->configuration['recommended_link_metrics'] = $values['metrics_settings']['recommended_link_metrics'];
    $this->configuration['recommended_external_link_metrics'] = $values['metrics_settings']['recommended_external_link_metrics'];
    $this->configuration['titles'] = $values['metrics_settings']['titles'];
    $this->configuration['recommended_title_metrics'] = $values['metrics_settings']['recommended_title_metrics'];
    $this->configuration['performance'] = $values['metrics_settings']['performance'];
    $this->configuration['classes'] = $values['metrics_settings']['classes'];
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $config = $this->getConfiguration();
    $settings['textClarityChecker'] = [
      'text_metrics' => $config['text_metrics'],
      'images' => $config['images'],
      'recommended_image_metrics' => $config['recommended_image_metrics'],
      'links' => $config['links'],
      'recommended_link_metrics' => $config['recommended_link_metrics'],
      'recommended_external_link_metrics' => $config['recommended_external_link_metrics'],
      'titles' => $config['titles'],
      'recommended_title_metrics' => $config['recommended_title_metrics'],
      'performance' => $config['performance'],
      'classes' => $config['classes'],
    ];

    return [
      '#theme' => 'text_clarity_checker',
      '#config' => $settings['textClarityChecker'],
      '#attached' => [
        'library' => ['text_clarity_checker/text_clarity_checker'],
        'drupalSettings' => [
          'textClarityChecker' => $settings['textClarityChecker'],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge(): int {
    return 0;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    return ['route'];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    return ['block:' . $this->getPluginId()];
  }

}
