<?php

namespace Drupal\text_overlay_image\Plugin\Block;

use Drupal\Core\Block\Attribute\Block;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\Entity\File;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a text overlay image block.
 */
#[Block(
  id: 'text_overlay_image_block',
  admin_label: new TranslatableMarkup('Text Overlay Image Block'),
  category: new TranslatableMarkup('Block'),
)]

class TextOverlayImageBlock extends BlockBase implements ContainerFactoryPluginInterface {

   /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected $fileUrlGenerator;

  /**
   * Constructs a new MyCustomBlock object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, FileUrlGeneratorInterface $file_url_generator) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fileUrlGenerator = $file_url_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_url_generator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'background_image' => '',
      'heading_title' => $this->t('TEXT OVERLAY IMAGE'),
      'subtext' => $this->t('with subtext'),
      'opacity' => 0.4,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {

    // Define the image upload form element.
    $form['background_image'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Background Image'),
      '#description' => $this->t('Upload an background image for this block.'),
      '#required' => TRUE,
      '#upload_location' => 'public://text_overlay_image/', // Directory to save the file.
      '#default_value' => $this->configuration['background_image'] ?? '',
      '#multiple' => FALSE,
    ];

    $form['heading_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Heading Title'),
      '#description' => $this->t('Heading title field.'),
      '#required' => TRUE,
      '#default_value' => $this->configuration['heading_title'] ?? '',
    ];

    $form['subtext'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subtext'),
      '#description' => $this->t('Subtext field.'),
      '#default_value' => $this->configuration['subtext'] ?? '',
    ];

    $form['opacity'] = [
      '#type' => 'number',
      '#title' => $this->t('Opacity value'),
      '#min' => 0.1,
      '#max' => 0.9,
      '#step' => 0.1,
      '#description' => $this->t('Opacity value field. From 0.1 to 0.9'),
      '#default_value' => $this->configuration['opacity'] ?? '',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {

    $this->configuration['background_image'] = $form_state->getValue('background_image');
    $this->configuration['heading_title'] = $form_state->getValue('heading_title');
    $this->configuration['subtext'] = $form_state->getValue('subtext');
    $this->configuration['opacity'] = $form_state->getValue('opacity');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {

    $file_id = $this->configuration['background_image'][0];
    if (empty($file_id)) return;
    
    $file = File::load($file_id);
    $image_url = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());

    return [
      '#theme' => 'text_overlay_image_block',
      '#url' => $image_url,
      '#heading_title' => $this->configuration['heading_title'],
      '#subtext' => $this->configuration['subtext'],
      '#opacity' => $this->configuration['opacity'],
    ];
  }

}
