<?php

namespace Drupal\textlink\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;

/**
 * Plugin implementation of the 'configurable_url_link' formatter.
 */
#[FieldFormatter(
  id: 'configurable_url_link',
  label: new TranslatableMarkup('Configurable URL Link'),
  field_types: ['string', 'text']
)]
class ConfigurableUrlLinkFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'base_url' => 'https://example.com/',
      'link_text' => '@value',
      'auto_sanitize' => FALSE,
      'sanitize_separator' => '-',
      'open_new_window' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);
    
    $elements['base_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Base URL'),
      '#default_value' => $this->getSetting('base_url'),
      '#description' => $this->t('The base URL that the field value will be appended to. Include trailing slash if needed (e.g., https://example.com/search/)'),
      '#required' => TRUE,
    ];
    
    $elements['link_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link text'),
      '#default_value' => $this->getSetting('link_text'),
      '#description' => $this->t('The text to display for the link. Use @value to include the field value, @sanitized for sanitized value, or enter custom text.'),
      '#required' => TRUE,
    ];
    
    $elements['auto_sanitize'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-sanitize field value for URL'),
      '#default_value' => $this->getSetting('auto_sanitize'),
      '#description' => $this->t('Automatically convert spaces to separators and make lowercase for URL generation (e.g., "Hello World" becomes "hello-world" in URL).'),
    ];
    
    $elements['sanitize_separator'] = [
      '#type' => 'select',
      '#title' => $this->t('Separator character'),
      '#default_value' => $this->getSetting('sanitize_separator'),
      '#options' => [
        '-' => $this->t('Hyphen (-)'),
        '_' => $this->t('Underscore (_)'),
        '.' => $this->t('Dot (.)'),
        '' => $this->t('Remove spaces entirely'),
      ],
      '#description' => $this->t('Character to replace spaces with when auto-sanitizing.'),
      '#states' => [
        'visible' => [
          ':input[name="fields[' . $this->fieldDefinition->getName() . '][settings_edit_form][settings][auto_sanitize]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    
    $elements['open_new_window'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Open link in new window'),
      '#default_value' => $this->getSetting('open_new_window'),
    ];
    
    // Add some example configurations
    $elements['examples'] = [
      '#type' => 'details',
      '#title' => $this->t('Configuration Examples'),
      '#open' => FALSE,
      '#description' => $this->t('<strong>LinkedIn Profile:</strong><br>
        Base URL: https://www.linkedin.com/in/<br>
        Link text: View @value on LinkedIn<br>
        Auto-sanitize: Checked<br><br>
        
        <strong>Search Engine:</strong><br>
        Base URL: https://www.google.com/search?q=<br>
        Link text: Search for "@value"<br>
        Auto-sanitize: Checked<br><br>
        
        <strong>Custom Page:</strong><br>
        Base URL: https://mysite.com/page/<br>
        Link text: @value<br>
        Auto-sanitize: Unchecked (if URLs are pre-formatted)'),
    ];
    
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function validateSettingsForm(array $form, FormStateInterface $form_state) {
    $base_url = $form_state->getValue(['fields', $this->fieldDefinition->getName(), 'settings_edit_form', 'settings', 'base_url']);
    
    if (!empty($base_url) && !filter_var($base_url, FILTER_VALIDATE_URL)) {
      $form_state->setError($form['base_url'], $this->t('Please enter a valid URL.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->t('Base URL: @url', ['@url' => $this->getSetting('base_url')]);
    $summary[] = $this->t('Link text: @text', ['@text' => $this->getSetting('link_text')]);
    
    if ($this->getSetting('auto_sanitize')) {
      $separator = $this->getSetting('sanitize_separator');
      $separator_text = $separator ?: 'remove spaces';
      $summary[] = $this->t('Auto-sanitize enabled (separator: @sep)', ['@sep' => $separator_text]);
    }
    
    if ($this->getSetting('open_new_window')) {
      $summary[] = $this->t('Opens in new window');
    }
    
    return $summary;
  }

  /**
   * Sanitizes a field value for URL usage.
   *
   * @param string $value
   *   The field value to sanitize.
   * @param string $separator
   *   The separator character to use.
   *
   * @return string
   *   The sanitized value.
   */
  protected function sanitizeValue($value, $separator = '-') {
    // Convert to lowercase
    $sanitized = strtolower($value);
    
    // Replace multiple spaces with single space
    $sanitized = preg_replace('/\s+/', ' ', $sanitized);
    
    // Replace spaces with separator (or remove if separator is empty)
    if ($separator !== '') {
      $sanitized = str_replace(' ', $separator, $sanitized);
      // Remove multiple consecutive separators
      $sanitized = preg_replace('/' . preg_quote($separator, '/') . '+/', $separator, $sanitized);
    } else {
      // Remove spaces entirely
      $sanitized = str_replace(' ', '', $sanitized);
    }
    
    // Remove any characters that aren't alphanumeric, separator, or basic punctuation
    $allowed_chars = 'a-z0-9._-';
    if ($separator !== '' && !in_array($separator, ['-', '_', '.'])) {
      $allowed_chars .= preg_quote($separator, '/');
    }
    $sanitized = preg_replace('/[^' . $allowed_chars . ']/', '', $sanitized);
    
    // Trim separators from beginning and end
    $sanitized = trim($sanitized, $separator . '._-');
    
    return $sanitized;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    
    foreach ($items as $delta => $item) {
      $field_value = trim($item->value);
      if (!empty($field_value)) {
        $original_value = $field_value;
        $url_value = $field_value;
        
        // Sanitize value for URL if enabled
        if ($this->getSetting('auto_sanitize')) {
          $separator = $this->getSetting('sanitize_separator');
          $url_value = $this->sanitizeValue($field_value, $separator);
        }
        
        // Build the full URL
        $base_url = $this->getSetting('base_url');
        $full_url = $base_url . ltrim($url_value, '/');
        
        // Prepare link text
        $link_text = $this->getSetting('link_text');
        
        // Replace placeholders in link text
        $link_text = str_replace('@value', $original_value, $link_text);
        $link_text = str_replace('@sanitized', $url_value, $link_text);
        
        // Prepare link attributes
        $attributes = [];
        if ($this->getSetting('open_new_window')) {
          $attributes['target'] = '_blank';
          $attributes['rel'] = 'noopener noreferrer';
        }
        
        $elements[$delta] = [
          '#type' => 'link',
          '#title' => $link_text,
          '#url' => Url::fromUri($full_url),
          '#attributes' => $attributes,
        ];
      }
    }
    
    return $elements;
  }
}