<?php

namespace Drupal\texts\Controller;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Drupal\Core\StreamWrapper\StreamWrapperManagerInterface;
use League\Csv\Writer;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;

/**
 * Controller for handling texts import/export operations.
 */
class TextsExportController extends ControllerBase {

  /**
   * Constructs a new TextsImportExportController.
   */
  public function __construct(
    LanguageManagerInterface $language_manager,
    protected FileSystemInterface $fileSystem,
    protected StreamWrapperManagerInterface $streamWrapperManager,
    EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('file_system'),
      $container->get('stream_wrapper_manager'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Export texts to CSV.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The CSV file response.
   */
  public function export() {
    $languages = $this->languageManager->getLanguages();
    $enabled_languages = array_filter($languages, function (LanguageInterface $language) {
      return $language->getId() !== LanguageInterface::LANGCODE_NOT_SPECIFIED
        && $language->getId() !== LanguageInterface::LANGCODE_NOT_APPLICABLE;
    });

    // Create CSV writer with UTF-8 encoding
    $csv = Writer::createFromString('');
    $csv->setDelimiter(';');
    $csv->setOutputBOM(Writer::BOM_UTF8);

    // Prepare headers
    $headers = ['full_key', 'type'];
    foreach ($enabled_languages as $language) {
      $headers[] = $language->getId();
    }
    $csv->insertOne($headers);

    // Get all texts using entity query
    $query = $this->entityTypeManager->getStorage('texts')->getQuery()
      ->accessCheck(FALSE)
      ->sort('key')
      ->sort('context');
    $ids = $query->execute();

    if (!empty($ids)) {
      $texts = $this->entityTypeManager->getStorage('texts')->loadMultiple($ids);

      // Organize translations by key and context
      $translations = [];
      foreach ($texts as $text) {
        /** @var TextsInterface $text */
        $key = $text->getTranslationKey();
        $context = $text->getContext();
        $full_key = $context . '.' . $key;
        $is_plural = $text->isPlural();

        if (!isset($translations[$full_key])) {
          $translations[$full_key] = [
            'is_plural' => $is_plural,
          ];
        }

        // Get translation for each language
        foreach ($enabled_languages as $language) {
          $langcode = $language->getId();
          if ($text->hasTranslation($langcode)) {
            $translation = $text->getTranslation($langcode);
            $translation_text = $translation->getTranslationText();

            if ($is_plural) {
              // For plural forms, split the translation into singular and plural
              $parts = explode(PoItem::DELIMITER, $translation_text);
              $translations[$full_key][$langcode] = [
                'singular' => $parts[0] ?? '',
                'plural' => $parts[1] ?? '',
              ];
            }
            else {
              $translations[$full_key][$langcode] = $translation_text;
            }
          }
        }
      }

      // Write translations to CSV
      foreach ($translations as $full_key => $data) {
        if ($data['is_plural']) {
          // For plural forms, create two rows - one for singular and one for plural
          $singular_row = [$full_key, 'singular'];
          $plural_row = [$full_key, 'plural'];

          foreach ($enabled_languages as $language) {
            $langcode = $language->getId();
            if (isset($data[$langcode])) {
              $singular_row[] = trim($data[$langcode]['singular']);
              $plural_row[] = trim($data[$langcode]['plural']);
            }
            else {
              $singular_row[] = '';
              $plural_row[] = '';
            }
          }
          $csv->insertOne($singular_row);
          $csv->insertOne($plural_row);
        }
        else {
          // For singular forms, create one row with type 'default'
          $row = [$full_key, 'default'];
          foreach ($enabled_languages as $language) {
            $langcode = $language->getId();
            if (isset($data[$langcode])) {
              $row[] = $data[$langcode];
            }
            else {
              $row[] = '';
            }
          }
          $csv->insertOne($row);
        }
      }
    }

    // Create response with UTF-8 encoding
    $response = new Response($csv->toString());
    $response->headers->set('Content-Type', 'text/csv; charset=utf-8');
    $response->headers->set('Content-Disposition', 'attachment; filename="texts_export.csv"');

    return $response;
  }

}
