<?php

namespace Drupal\texts\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\texts\Entity\Texts;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Return response for manual check translations.
 */
class TextsController extends ControllerBase {

  /**
   * Shows the string locale search screen.
   *
   * @return array
   *   The render array for the string search screen.
   */
  public function translatePage() {
    return [
      'filter' => $this->formBuilder()->getForm('Drupal\texts\Form\Overview\TranslationFilterForm'),
      'form' => $this->formBuilder()->getForm('Drupal\texts\Form\Overview\TranslationEditForm'),
    ];
  }

  /**
   * Restores a duplicate entity by copying its translation to the original key.
   *
   * @param \Drupal\texts\Entity\Texts $texts
   *   The duplicate texts entity.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response.
   */
  public function restoreDuplicate(Texts $texts) {
    $key = $texts->getTranslationKey();
    if (!preg_match('/^(.+)__duplicate_\d+$/', $key, $matches)) {
      $this->messenger()->addError($this->t('This entity does not appear to be a duplicate.'));
      return new RedirectResponse($texts->toUrl('collection')->toString());
    }

    $original_key = $matches[1];
    $context = $texts->getContext();

    // Find the original entity by key and context.
    $storage = $this->entityTypeManager()->getStorage('texts');
    $query = $storage->getQuery()
      ->accessCheck(FALSE)
      ->condition('key', $original_key)
      ->condition('context', $context);

    $ids = $query->execute();

    if (empty($ids)) {
      $this->messenger()->addError($this->t('Original entity with key "@key" and context "@context" not found.', [
        '@key' => $original_key,
        '@context' => $context,
      ]));
      return new RedirectResponse($texts->toUrl('collection')->toString());
    }

    $original_id = reset($ids);
    $original_entity = $storage->load($original_id);

    if (!$original_entity) {
      $this->messenger()->addError($this->t('Could not load original entity.'));
      return new RedirectResponse($texts->toUrl('collection')->toString());
    }

    try {
      // Get all available translations from the duplicate entity.
      $duplicate_translations = $texts->getTranslationLanguages();
      $restored_count = 0;

      // Copy all translations from duplicate to original.
      foreach ($duplicate_translations as $langcode => $language) {
        $duplicate_translation = $texts->getTranslation($langcode);

        // Get or create the corresponding translation in the original entity.
        if ($original_entity->hasTranslation($langcode)) {
          $original_translation = $original_entity->getTranslation($langcode);
        }
        else {
          $original_translation = $original_entity->addTranslation($langcode);
        }

        // Copy translation data from duplicate to original.
        $original_translation->setTranslationText($duplicate_translation->getTranslationText());
        $original_translation->setPlural($duplicate_translation->isPlural());
        $original_translation->save();
        $restored_count++;
      }

      // Delete the duplicate entity after all translations are restored.
      $texts->delete();

      $this->messenger()->addStatus($this->t('Successfully restored @count translation(s) from duplicate to original key "@key".', [
        '@count' => $restored_count,
        '@key' => $original_key,
      ]));

      // Redirect to the original entity edit form or back to collection.
      $request = \Drupal::request();
      $destination = $request->query->get('destination');
      if ($destination) {
        return new RedirectResponse($destination);
      }
      return new RedirectResponse($original_entity->toUrl('edit-form')->toString());
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Error restoring translation: @message', [
        '@message' => $e->getMessage(),
      ]));
      $this->logger('texts')->error('Error restoring duplicate entity: @message', [
        '@message' => $e->getMessage(),
      ]);
      return new RedirectResponse($texts->toUrl('collection')->toString());
    }
  }

}
