<?php

namespace Drupal\texts\Drush\Commands;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Drush commands for the Texts module.
 */
class TextsCommands extends DrushCommands {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a TextsCommands object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct();
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('database'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Find and backup duplicate texts entities with the same key+context+langcode.
   *
   * This command identifies texts entities that have duplicate key+context+langcode
   * combinations and renames the duplicates with a __duplicate_{id} suffix,
   * keeping the oldest entity (lowest ID) with the original key. The same
   * key+context combination can exist for different languages (translations).
   * The renamed entities can be restored later via the admin form.
   *
   * @param array $options
   *   Command options.
   *
   * @command texts:cleanup-duplicates
   * @aliases texts-cleanup
   * @usage texts:cleanup-duplicates
   *   Find and backup all duplicate texts entities by renaming their keys.
   * @usage texts:cleanup-duplicates --dry-run
   *   Preview duplicates without renaming them.
   */
  #[CLI\Command(name: 'texts:cleanup-duplicates', aliases: ['texts-cleanup'])]
  #[CLI\Option(name: 'dry-run', description: 'Preview duplicates without renaming them.')]
  public function cleanupDuplicates(array $options = ['dry-run' => FALSE]): void {
    $dry_run = $options['dry-run'];

    // Find all duplicate key+context+langcode combinations, excluding already renamed ones.
    $duplicates = $this->database->query("
      SELECT `key`, context, langcode, COUNT(*) as count, MIN(id) as keep_id
      FROM {texts_field_data}
      WHERE `key` NOT LIKE '%__duplicate_%'
      GROUP BY `key`, context, langcode
      HAVING COUNT(*) > 1
    ")->fetchAll();

    if (empty($duplicates)) {
      $this->logger()->success('No duplicate texts entities found.');
      return;
    }

    $total_duplicates = 0;
    $total_to_rename = 0;
    $storage = $this->entityTypeManager->getStorage('texts');

    foreach ($duplicates as $dup) {
      $total_duplicates++;
      $to_rename_count = $dup->count - 1;
      $total_to_rename += $to_rename_count;

      $this->logger()->notice(sprintf(
        'Found %d duplicates for key="%s", context="%s", langcode="%s" (keeping ID %d)',
        $dup->count,
        $dup->key,
        $dup->context,
        $dup->langcode,
        $dup->keep_id
      ));

      if (!$dry_run) {
        // Find all IDs except the one to keep.
        $ids_to_rename = $this->database->query("
          SELECT id FROM {texts_field_data}
          WHERE `key` = :key AND context = :context AND langcode = :langcode AND id != :keep_id
        ", [
          ':key' => $dup->key,
          ':context' => $dup->context,
          ':langcode' => $dup->langcode,
          ':keep_id' => $dup->keep_id,
        ])->fetchCol();

        if (!empty($ids_to_rename)) {
          $entities = $storage->loadMultiple($ids_to_rename);
          $renamed_count = 0;
          foreach ($entities as $entity) {
            /** @var \Drupal\texts\TextsInterface $entity */
            $original_key = $entity->getTranslationKey();
            $new_key = $original_key . '__duplicate_' . $entity->id();
            $entity->setTranslationKey($new_key);
            $entity->save();
            $renamed_count++;
            $this->logger()->info(sprintf(
              'Renamed duplicate entity ID %d: "%s" -> "%s"',
              $entity->id(),
              $original_key,
              $new_key
            ));
          }
          $this->logger()->info(sprintf(
            'Renamed %d duplicate entities for key="%s", context="%s", langcode="%s"',
            $renamed_count,
            $dup->key,
            $dup->context,
            $dup->langcode
          ));
        }
      }
    }

    if ($dry_run) {
      $this->logger()->warning(sprintf(
        'DRY RUN: Found %d key+context+langcode combinations with duplicates (%d entities to rename). Run without --dry-run to rename them.',
        $total_duplicates,
        $total_to_rename
      ));
    }
    else {
      $this->logger()->success(sprintf(
        'Cleaned up %d duplicate key+context+langcode combinations, renamed %d entities with __duplicate_{id} suffix.',
        $total_duplicates,
        $total_to_rename
      ));
    }
  }

  /**
   * Delete all texts entities with __duplicate_ suffix in their key.
   *
   * This command identifies and permanently deletes all texts entities that
   * have been marked as duplicates (keys ending with __duplicate_{id}).
   * Use this command to clean up old duplicate backups that are no longer
   * needed.
   *
   * @param array $options
   *   Command options.
   *
   * @command texts:delete-duplicates
   * @aliases texts-delete-dups
   * @usage texts:delete-duplicates
   *   Delete all duplicate texts entities.
   * @usage texts:delete-duplicates --dry-run
   *   Preview duplicates without deleting them.
   */
  #[CLI\Command(name: 'texts:delete-duplicates', aliases: ['texts-delete-dups'])]
  #[CLI\Option(name: 'dry-run', description: 'Preview duplicates without deleting them.')]
  public function deleteDuplicates(array $options = ['dry-run' => FALSE]): void {
    $dry_run = $options['dry-run'];

    // Find all entities with __duplicate_ suffix in their key.
    $duplicate_ids = $this->database->query("
      SELECT id FROM {texts_field_data}
      WHERE `key` LIKE '%__duplicate_%'
    ")->fetchCol();

    if (empty($duplicate_ids)) {
      $this->logger()->success('No duplicate texts entities found.');
      return;
    }

    $storage = $this->entityTypeManager->getStorage('texts');
    $entities = $storage->loadMultiple($duplicate_ids);
    $count = count($entities);

    $this->logger()->notice(sprintf(
      'Found %d duplicate entities to delete.',
      $count
    ));

    if (!$dry_run) {
      $deleted_count = 0;
      foreach ($entities as $entity) {
        /** @var \Drupal\texts\TextsInterface $entity */
        $entity_id = $entity->id();
        $key = $entity->getTranslationKey();
        $context = $entity->getContext();
        $entity->delete();
        $deleted_count++;
        $this->logger()->info(sprintf(
          'Deleted duplicate entity ID %d with key "%s" (context: %s)',
          $entity_id,
          $key,
          $context
        ));
      }

      $this->logger()->success(sprintf(
        'Successfully deleted %d duplicate entities.',
        $deleted_count
      ));
    }
    else {
      $this->logger()->warning(sprintf(
        'DRY RUN: Found %d duplicate entities that would be deleted. Run without --dry-run to delete them.',
        $count
      ));

      // Show some examples of what would be deleted.
      $examples = array_slice($entities, 0, 5);
      foreach ($examples as $entity) {
        /** @var \Drupal\texts\TextsInterface $entity */
        $this->logger()->info(sprintf(
          '  - ID %d: "%s" (context: %s)',
          $entity->id(),
          $entity->getTranslationKey(),
          $entity->getContext()
        ));
      }
      if ($count > 5) {
        $this->logger()->info(sprintf('  ... and %d more', $count - 5));
      }
    }
  }

}
