<?php

namespace Drupal\texts\Form;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use League\Csv\Reader;
use Symfony\Component\HttpFoundation\File\UploadedFile;

/**
 * Provides a form for importing texts.
 */
class TextsImportForm extends FormBase {

  /**
   * The texts storage.
   *
   * @var \Drupal\texts\TextsStorage
   */
  protected $textsStorage;

  /**
   * Constructs a new TextsImportForm.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $keyValueFactory
   *   The key-value store factory.
   */
  public function __construct(
    protected LanguageManagerInterface $languageManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected KeyValueFactoryInterface $keyValueFactory,
  ) {
    $this->textsStorage = $entityTypeManager->getStorage('texts');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('language_manager'),
      $container->get('entity_type.manager'),
      $container->get('keyvalue'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'texts_import_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['file'] = [
      '#type' => 'file',
      '#title' => $this->t('CSV file'),
      '#description' => $this->t('Select a CSV file to import.'),
      '#upload_validators' => [
        'file_validate_extensions' => ['csv'],
      ],
      '#upload_location' => 'temporary://',
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Import'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $all_files = $this->getRequest()->files->get('files', []);
    if (empty($all_files['file'])) {
      $form_state->setErrorByName('file', $this->t('Please select a file to upload.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $all_files = $this->getRequest()->files->get('files', []);
    if (!empty($all_files['file'])) {
      $file = $all_files['file'];
      if (!$file instanceof UploadedFile || !$file->isValid()) {
        $this->messenger()->addError($this->t('The uploaded file is not valid.'));
        return;
      }

      // Store the data from the file in the key-value store so it can be
      // processed by any available Drupal instance.
      $content = $file->getContent();
      if (empty($content)) {
        $this->messenger()->addWarning($this->t('The uploaded file is empty.'));
        return;
      }

      $key = (string) time();
      $key_value_store = $this->keyValueFactory->get('texts.csv_import');
      $key_value_store->set($key, $content);

      // Create a batch operation
      $batch_builder = new BatchBuilder();
      $batch_builder
        ->setTitle($this->t('Importing texts...'))
        ->setInitMessage($this->t('Starting import...'))
        ->setProgressMessage($this->t('Processed @current out of @total.'))
        ->setErrorMessage($this->t('An error occurred during import.'))
        ->addOperation([$this, 'prepareImportBatch'], [$key])
        ->addOperation([$this, 'processImportBatch'])
        ->setFinishCallback([$this, 'finishImportBatch']);

      batch_set($batch_builder->toArray());
    }
  }

  /**
   * Prepares the import batch operation.
   *
   * Transforms the uploaded file content into a format suitable for processing.
   *
   * @param string $key
   *   The key under which the file content is stored in the key-value store.
   * @param array $context
   *   The batch context.
   */
  public function prepareImportBatch($key, array &$context) {
    $key_value_store = $this->keyValueFactory->get('texts.csv_import');
    $file_content = $key_value_store->get($key, '');

    try {
      $csv = Reader::createFromString($file_content);
      $csv->setDelimiter(';');
      $csv->setHeaderOffset(0);

      // Get header row to determine languages
      $header = $csv->getHeader();
      if (empty($header)) {
        $context['message'] = $this->t('Could not read CSV header.');
        return;
      }

      // Skip full_key and type columns, rest are language codes
      $languages = array_slice($header, 2);
      // Filter out default language
      $default_langcode = $this->languageManager->getDefaultLanguage()->getId();
      $translation_languages = array_filter($languages, function ($langcode) use ($default_langcode) {
        return $this->languageManager->getLanguage($langcode) && $langcode !== $default_langcode;
      });

      $context['results']['default_langcode'] = $default_langcode;
      $context['results']['translation_languages'] = $translation_languages;

      // We have to process the whole CSV file to get the total number of keys to process
      // because singular / plural data is spread over multiple rows.
      $records = $csv->getRecords($header);
      foreach ($records as $row) {
        if (count($row) < 3) {
          continue;
        }

        $full_key = $row['full_key'];
        $type = $row['type'];

        if (!isset($context['results']['data'][$full_key])) {
          $context['results']['data'][$full_key] = [];
          $context['results']['keys'][] = $full_key;
        }

        $context['results']['data'][$full_key][$type] = array_combine($languages, array_slice($row, 2));
      }

      // Set the total number of keys to process
      $context['results']['max'] = count($context['results']['keys']);
    }
    catch (\Exception $e) {
      $context['message'] = $this->t('An error occurred while preparing the import: @error', ['@error' => $e->getMessage()]);
    }

    // Clean up the source data. It is no longer needed.
    $key_value_store->deleteAll();
  }

  /**
   * Process the import batch operation.
   *
   * @param array $context
   *   The batch context.
   */
  public function processImportBatch(array &$context) {
    if (!isset($context['sandbox']['progress'])) {
      $context['sandbox']['progress'] = 0;
    }

    $batch_size = 50;
    $translation_languages = $context['results']['translation_languages'] ?? [];
    $default_langcode = $context['results']['default_langcode'] ?? '';

    try {
      // Process the next batch of keys
      $keys_to_process = array_slice($context['results']['keys'], $context['sandbox']['progress'], $batch_size);
      foreach ($keys_to_process as $full_key) {
        $types = $context['results']['data'][$full_key];
        $context['sandbox']['progress']++;

        // Process the translations
        // Split full_key into context and key
        [$translation_context, $key] = explode('.', $full_key, 2);

        // Check if text entity exists
        $text = $this->textsStorage->loadByKey($key, $translation_context);

        if (!$text) {
          // Create new entity for both default and singular/plural types
          if (isset($types['default']) || isset($types['singular']) || isset($types['plural'])) {
            /** @var \Drupal\texts\TextsInterface $text */
            $text = $this->textsStorage->create([
              'key' => $key,
              'context' => $translation_context,
              'translation' => '',
            ]);

            // Set default language translation if available
            if (isset($types['default'][$default_langcode])) {
              $text->set('translation', $types['default'][$default_langcode]);
            }
            elseif (isset($types['singular'][$default_langcode])) {
              $plural = '';
              if (!empty($types['plural'][$default_langcode])) {
                $plural = $types['plural'][$default_langcode];
              }
              $text->set('translation', implode(PoItem::DELIMITER, [$types['singular'][$default_langcode], $plural]));
              $text->set('plural', TRUE);
            }

            // The entity->save() function does not execute the UniqueValue constraint.
            $violations = $text->validate();
            if (count($violations) === 0) {
              $text->save();
            }

            // Set translations for all languages
            foreach ($translation_languages as $langcode) {
              if (!empty($types['default'][$langcode])) {
                $text->addTranslation($langcode, [
                  'translation' => $types['default'][$langcode],
                ]);
              }
            }

            // Handle singular/plural types
            if (isset($types['singular']) || isset($types['plural'])) {
              $text->set('plural', TRUE);

              foreach ($translation_languages as $langcode) {
                if (!empty($types['singular'][$langcode])) {
                  if (!$text->hasTranslation($langcode)) {
                    $text->addTranslation($langcode, [
                      'translation' => '',
                    ]);
                  }
                  $translation = $text->getTranslation($langcode);
                  $plural = $types['plural'][$langcode] ?? '';
                  $translation->set('translation', implode(PoItem::DELIMITER, [$types['singular'][$langcode], $plural]));
                }
              }
            }

            // The entity->save() function does not execute the UniqueValue constraint.
            $violations = $text->validate();
            if (count($violations) === 0) {
              $text->save();
            }
          }
        }
        else {
          // Update existing entity
          foreach ($types as $type => $values) {
            foreach ($values as $langcode => $value) {

              if (empty($value)) {
                // Skip empty values
                continue;
              }

              // Add a translation if it does not exist
              if (in_array($langcode, $translation_languages)) {
                if (!$text->hasTranslation($langcode)) {
                  $text->addTranslation($langcode, [
                    'translation' => '',
                  ]);
                }
              }

              $translation = $text->getTranslation($langcode);

              switch ($type) {
                case 'default':
                  // Default type sets translation and marks as non-plural
                  $translation->set('translation', $value);
                  break;

                case 'singular':
                  // We save the singular value and the plural value at the same time and save on entity save.
                  $plural_translation = implode(PoItem::DELIMITER, [$types['singular'][$langcode], $types['plural'][$langcode]]);
                  $translation->set('translation', $plural_translation);
                  $text->set('plural', TRUE);
                  break;
              }
            }

            // The entity->save() function does not execute the UniqueValue constraint.
            $violations = $text->validate();
            if (count($violations) === 0) {
              $text->save();
            }
          }
        }
      }

      if ($context['sandbox']['progress'] != $context['results']['max']) {
        $context['finished'] = $context['sandbox']['progress'] / $context['results']['max'];
      }
      else {
        $context['finished'] = 1;
      }
    }
    catch (\Exception $e) {
      $context['message'] = $this->t('An error occurred during import: @error', ['@error' => $e->getMessage()]);
    }
  }

  /**
   * Finish the import batch operation.
   *
   * @param bool $success
   *   Whether the batch operation was successful.
   * @param array $results
   *   The results of the batch operation.
   * @param array $operations
   *   The operations that were performed.
   */
  public function finishImportBatch($success, array $results, array $operations) {
    if ($success) {
      $this->messenger()->addStatus($this->t('Texts have been imported successfully.'));
    }
    else {
      $this->messenger()->addError($this->t('An error occurred during import.'));
    }
  }

}
