<?php

namespace Drupal\tfa;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\tfa\Attribute\Tfa;

/**
 * Tfa plugin manager.
 *
 * @internal
 */
class TfaPluginManager extends DefaultPluginManager {

  const string PLUGIN_LOGIN = "login_plugins";
  const string PLUGIN_SEND = "send_plugins";
  const string PLUGIN_VALIDATION = "allowed_validation_plugins";

  /**
   * Constructs TfaPluginManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler, protected ConfigFactoryInterface $configFactory) {
    parent::__construct(
      'Plugin/Tfa',
      $namespaces,
      $module_handler,
      'Drupal\tfa\TfaPluginInterface',
      Tfa::class,
    );
    $this->alterInfo('tfa_info');
    $this->setCacheBackend($cache_backend, 'tfa_plugins');
  }

  /**
   * Return plugin definitions for all validation plugins.
   *
   * @param bool $active_only
   *   Whether to only get active plugins enabled in config.
   *
   * @return array
   *   Array of validation plugin definitions.
   */
  public function getValidationDefinitions(bool $active_only = TRUE): array {
    return $this->getPluginsByTypeAndStatus(self::PLUGIN_VALIDATION, $active_only);
  }

  /**
   * Return plugin definitions for all login plugins.
   *
   * @param bool $active_only
   *   Whether to only get active plugins enabled in config.
   *
   * @return array
   *   Array of login plugin definitions.
   */
  public function getLoginDefinitions(bool $active_only = TRUE): array {
    return $this->getPluginsByTypeAndStatus(self::PLUGIN_LOGIN, $active_only);
  }

  /**
   * Return plugin definitions for all send plugins.
   *
   * @param bool $active_only
   *   Whether to only get active plugins enabled in config.
   *
   * @return array
   *   Array of send plugin definitions.
   */
  public function getSendDefinitions(bool $active_only = TRUE): array {
    return $this->getPluginsByTypeAndStatus(self::PLUGIN_SEND, $active_only);
  }

  /**
   * Return plugin definitions for all plugins implementing the given class.
   *
   * @param string $class
   *   Name of the class or interface that the plugins must implement.
   *
   * @return array
   *   Array of plugin definitions.
   */
  public function getClassDefinitions(string $class): array {
    $all_plugins = $this->getDefinitions();
    $plugins = [];

    foreach ($all_plugins as $key => $plugin) {
      if (is_a($plugin['class'], $class, TRUE)) {
        $plugins[$key] = $plugin;
      }
    }
    return $plugins;
  }

  /**
   * Get plugins by type.
   */
  protected function getPluginsByTypeAndStatus(string $type, bool $active_only): array {
    $plugin_classmap = [
      self::PLUGIN_LOGIN => '\Drupal\tfa\Plugin\TfaLoginInterface',
      self::PLUGIN_SEND => '\Drupal\tfa\Plugin\TfaSendInterface',
      self::PLUGIN_VALIDATION => '\Drupal\tfa\Plugin\TfaValidationInterface',
    ];

    if (!array_key_exists($type, $plugin_classmap)) {
      return [];
    }

    $plugins = $this->getClassDefinitions($plugin_classmap[$type]);
    if ($active_only) {
      $enabled_plugins = $this->configFactory->get('tfa.settings')->get($type);
      if (!is_array($enabled_plugins)) {
        return [];
      }

      $plugins = array_filter($plugins, function (array $plugin) use ($enabled_plugins) {
        return ($enabled_plugins[$plugin['id']] ?? NULL) === $plugin['id'];
      });

    }
    return $plugins;
  }

}
