<?php

namespace Drupal\Tests\tfa\Functional;

use Drupal\tfa\Plugin\Tfa\TfaRecoveryCode;
use Drupal\tfa\TfaPluginManager;
use Drupal\user\UserInterface;

/**
 * Class TfaRecoveryCodeSetupPluginTest.
 *
 * @group tfa
 *
 * @ingroup tfa
 */
class TfaRecoveryCodePluginTest extends TfaTestBase {

  /**
   * Plugin id for the recovery code validation plugin.
   *
   * @var string
   */
  protected string $validationPluginId = 'tfa_recovery_code';

  /**
   * Non-admin user account. Standard tfa user.
   *
   * @var \Drupal\user\UserInterface
   */
  public UserInterface $userAccount;

  /**
   * Tfa plugin manager.
   *
   * @var \Drupal\tfa\TfaPluginManager
   */
  public TfaPluginManager $tfaValidationManager;

  /**
   * Instance of the validation plugin for the $validationPluginId.
   *
   * @var \Drupal\tfa\Plugin\Tfa\TfaRecoveryCode
   */
  public TfaRecoveryCode $validationPlugin;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'tfa_test_user',
  ];

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $config = $this->config('tfa.settings');
    $config->set('enabled', TRUE)
      ->set('default_validation_plugin', $this->validationPluginId)
      ->set('allowed_validation_plugins', [$this->validationPluginId => $this->validationPluginId])
      ->set('encryption', $this->encryptionProfile->id())
      ->set('required_roles', ['authenticated' => 'authenticated'])
      ->set('validation_plugin_settings', [
        $this->validationPluginId => [
          'recovery_codes_amount' => 10,
        ],
      ])
      ->save();

    $permissions = ['setup own tfa', 'disable own tfa'];
    $user_account = $this->createUser($permissions);
    assert($user_account !== FALSE);
    $this->userAccount = $user_account;

    $this->tfaValidationManager = \Drupal::service('plugin.manager.tfa');
    $validation_plugin = $this->tfaValidationManager->createInstance($this->validationPluginId, ['uid' => $this->userAccount->id()]);
    assert($validation_plugin instanceof TfaRecoveryCode);
    $this->validationPlugin = $validation_plugin;
  }

  /**
   * Test that we can enable the plugin.
   */
  public function testEnableValidationPlugin(): void {
    $this->canEnableValidationPlugin($this->validationPluginId);
  }

  /**
   * Check that recovery code plugin appear on the user overview page.
   */
  public function testRecoveryCodeOverviewExists(): void {
    $this->drupalLogin($this->userAccount);
    $this->drupalGet('user/' . $this->userAccount->id() . '/security/tfa');
    $assert = $this->assertSession();
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Recovery Codes');
  }

  /**
   * Check that the user can setup recovery codes.
   */
  public function testRecoveryCodeSetup(): void {
    $this->drupalLogin($this->userAccount);
    $this->drupalGet('user/' . $this->userAccount->id() . '/security/tfa/' . $this->validationPluginId . '/1');
    $assert = $this->assertSession();
    $assert->statusCodeEquals(200);
    $assert->responseContains('Enter your current password');

    // Provide the user's password to continue.
    $edit = ['current_pass' => $this->userAccount->passRaw];
    $this->submitForm($edit, 'Confirm');

    $assert->responseContains('Save codes to account');
    $this->submitForm([], 'Save codes to account');
    $assert->pageTextContains('TFA setup complete.');

    // Make sure codes were saved to the account.
    $codes = $this->validationPlugin->getCodes();
    $this->assertTrue(!empty($codes), 'No codes saved to the account data.');

    // Now the user should be able to see their existing codes. Let's test that.
    $assert->linkExists('Show codes');
    $this->drupalGet('user/' . $this->userAccount->id() . '/security/tfa/' . $this->validationPluginId);

    $edit = ['current_pass' => $this->userAccount->passRaw];
    $this->submitForm($edit, 'Confirm');
    $assert->statusCodeEquals(200);
    // The "save" button should not exists when viewing existing codes.
    $assert->responseNotContains('Save codes to account');
  }

  /**
   * Check that the user can login with recovery codes.
   */
  public function testRecoveryCodeValidation(): void {
    // Login the user, generate and save some codes, then log back out.
    $this->drupalLogin($this->userAccount);
    $assert = $this->assertSession();

    $codes = $this->validationPlugin->generateCodes();
    $this->validationPlugin->storeCodes($codes);
    $this->drupalLogout();

    // Password form.
    $edit = [
      'name' => $this->userAccount->getAccountName(),
      'pass' => $this->userAccount->passRaw,
    ];
    $this->drupalGet('user/login');
    $this->submitForm($edit, 'Log in');
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Enter one of your recovery codes');

    // Try an invalid code.
    $edit = ['code' => 'definitely not real'];
    $this->submitForm($edit, 'Verify');
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Invalid recovery code.');

    // Try a valid code.
    $edit['code'] = $codes[0];
    $this->submitForm($edit, 'Verify');
    $assert->statusCodeEquals(200);
    $assert->pageTextContains($this->userAccount->getDisplayName());
    $this->assertTrue($this->userAccount->isAuthenticated(), 'User is logged in.');

    // Try replay attack with a valid code that has already been used.
    $this->drupalLogout();
    $edit = [
      'name' => $this->userAccount->getAccountName(),
      'pass' => $this->userAccount->passRaw,
    ];
    $this->drupalGet('user/login');
    $this->submitForm($edit, 'Log in');
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Enter one of your recovery codes');

    $edit = ['code' => $codes[0]];
    $this->submitForm($edit, 'Verify');
    $assert->statusCodeEquals(200);
    $assert->pageTextContains('Invalid recovery code.');
  }

}
