<?php

namespace Drupal\tfa_email_support\EventSubscriber;

use Drupal\Core\Routing\RouteSubscriberBase;
use Symfony\Component\Routing\RouteCollection;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Drupal\Core\Session\AccountProxyInterface;
use Psr\Log\LoggerInterface;

/**
 * Route subscriber for TFA email support.
 */
class TfaEmailRouteSubscriber extends RouteSubscriberBase {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs a new TfaEmailRouteSubscriber.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(AccountProxyInterface $current_user, LoggerInterface $logger) {
    $this->currentUser = $current_user;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection) {
    // No route alterations currently required.
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[KernelEvents::REQUEST][] = ['onKernelRequest', 100];
    return $events;
  }

  /**
   * Handles the kernel request event.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event.
   */
  public function onKernelRequest(RequestEvent $event) {
    $request = $event->getRequest();
    $route_name = $request->attributes->get('_route');

    // Only process TFA-related routes.
    if ($route_name === null || strpos($route_name, 'tfa') === FALSE) {
      return;
    }

    $this->logger->notice('TFA route accessed: @route', ['@route' => $route_name]);

    // Extract UID from route parameters.
    $uid = null;
    if ($request->attributes->has('uid')) {
      $uid = $request->attributes->get('uid');
    }
    elseif ($request->attributes->has('user')) {
      $user = $request->attributes->get('user');
      $uid = is_object($user) ? $user->id() : $user;
    }

    // Store UID in request and session if found.
    if ($uid) {
      $request->attributes->set('tfa_uid', $uid);
      $this->logger->notice('TFA UID set in request: @uid', ['@uid' => $uid]);

      if ($request->hasSession()) {
        $session = $request->getSession();
        $tfa_data = $session->get('tfa', []);
        $tfa_data['uid'] = $uid;
        $session->set('tfa', $tfa_data);
      }
    }
  }

}
