<?php

namespace Drupal\tfa_email_support\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to enable or disable TFA via Email for a user.
 */
class EnableTfaEmailForm extends FormBase {

  protected $messenger;
  protected $entityTypeManager;
  protected $userData;
  protected $uid;

  public function __construct(
    MessengerInterface $messenger,
    EntityTypeManagerInterface $entity_type_manager,
    UserDataInterface $user_data
  ) {
    $this->messenger = $messenger;
    $this->entityTypeManager = $entity_type_manager;
    $this->userData = $user_data;
  }

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('messenger'),
      $container->get('entity_type.manager'),
      $container->get('user.data')
    );
  }

  public function getFormId() {
    return 'tfa_email_support_enable_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state, $uid = NULL) {
    $this->uid = $uid;

    if (empty($uid)) {
      $this->messenger->addError($this->t('No user ID provided.'));
      return [];
    }

    $user = $this->entityTypeManager->getStorage('user')->load($uid);
    if (!$user) {
      $this->messenger->addError($this->t('User not found.'));
      return [];
    }

    // Check TFA configuration.
    $config = \Drupal::configFactory()->get("tfa.settings.user.{$uid}");
    $validation_plugins = $config->get('validation_plugins') ?: [];
    $is_configured_in_config = in_array('tfa_email_support', $validation_plugins);

    $connection = \Drupal::database();
    
    // Check for tfa_user_settings in users_data
    $tfa_user_settings = $connection->select('users_data', 'ud')
      ->fields('ud', ['value'])
      ->condition('uid', $uid)
      ->condition('module', 'tfa')
      ->condition('name', 'tfa_user_settings')
      ->execute()
      ->fetchField();
    
    // Check for tfa_email_support in users_data
    $tfa_email_support = $connection->select('users_data', 'ud')
      ->fields('ud', ['value'])
      ->condition('uid', $uid)
      ->condition('module', 'tfa')
      ->condition('name', 'tfa_email_support')
      ->execute()
      ->fetchField();

    $email_plugin_enabled = FALSE;
    if ($tfa_email_support) {
      $tfa_email_support_data = unserialize($tfa_email_support, ['allowed_classes' => false]);
      if (is_array($tfa_email_support_data)) {
        $email_plugin_enabled = !empty($tfa_email_support_data['status']);
      } else {
        $email_plugin_enabled = ($tfa_email_support_data == 1);
      }
    }
if (!$email_plugin_enabled) {
  $editable_config = \Drupal::configFactory()->getEditable("tfa.settings.user.{$uid}");
  if (!$editable_config->isNew()) {
    $editable_config->delete();
  }
}
   $is_configured = $is_configured_in_config && $email_plugin_enabled;

    $tfa_email = $config->get('data.tfa_email_support.email') ?? $user->getEmail();

    $form['info'] = [
      '#markup' => $this->t('TFA via Email is currently: <strong>@status</strong> for user <em>@email</em>.', [
        '@status' => $is_configured ? $this->t('Enabled') : $this->t('Disabled'),
        '@email' => $tfa_email,
      ]),
    ];

    $form['tfa_email'] = [
      '#type' => 'email',
      '#title' => $this->t('TFA Email Address'),
      '#default_value' => $tfa_email,
      '#description' => $this->t('Enter the email address to be used for TFA codes.'),
      '#required' => TRUE,
    ];

    $form['action'] = [
      '#type' => 'submit',
      '#value' => $is_configured ? $this->t('Disable TFA Email') : $this->t('Enable TFA Email'),
      '#name' => $is_configured ? 'disable' : 'enable',
    ];

    $form['save_email'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update TFA Email'),
      '#name' => 'update_email',
    ];

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {
    if (empty($this->uid)) {
      $this->messenger->addError($this->t('No user ID provided.'));
      return;
    }

    $new_email = $form_state->getValue('tfa_email');
    $current_time = (int) \Drupal::time()->getRequestTime();

    $triggering_element = $form_state->getTriggeringElement();
    $clicked_name = $triggering_element['#name'] ?? '';

    if ($clicked_name === 'disable') {
      $this->disableEmailTfa();
    }
    elseif ($clicked_name === 'enable') {
      $this->enableEmailTfa($new_email, $current_time);
    }
    elseif ($clicked_name === 'update_email') {
      $this->updateTfaEmail($new_email, $current_time);
    }

    $this->clearTfaCaches();
    
    // Set rebuild to refresh the form with updated values
    $form_state->setRebuild(TRUE);
  }

  /**
   * Update only the TFA email.
   */
  protected function updateTfaEmail($email, $current_time) {
    $config = \Drupal::configFactory()->getEditable("tfa.settings.user.{$this->uid}");
    $plugin_data = $config->get('data.tfa_email_support') ?: [];
    $plugin_data['email'] = $email;
    $plugin_data['saved'] = $current_time;
    $config->set('data.tfa_email_support', $plugin_data);
    $config->save();

    // Set serialized value 1 for tfa_email_support in users_data
    $this->setUsersDataValue($this->uid, 'tfa', 'tfa_email_support', 1);

    $this->messenger->addStatus($this->t('TFA email updated to @mail.', ['@mail' => $email]));
  }


  /**
 * Enable TFA Email and set serialized value 1 for both tfa_email_support and tfa_user_settings.
 */
protected function enableEmailTfa($email, $current_time) {
  // Add tfa_email_support record
  $this->setUsersDataValue($this->uid, 'tfa', 'tfa_email_support', [
    'status' => 1,
    'saved' => $current_time,
    'email' => $email,
  ]);

  // Check if tfa_user_settings already exists
  $connection = \Drupal::database();
  $existing_tfa_user_settings = $connection->select('users_data', 'ud')
    ->fields('ud', ['value'])
    ->condition('uid', $this->uid)
    ->condition('module', 'tfa')
    ->condition('name', 'tfa_user_settings')
    ->execute()
    ->fetchField();

  if ($existing_tfa_user_settings) {
    // Update existing structure
    $tfa_user_settings = unserialize($existing_tfa_user_settings, ['allowed_classes' => false]);
    
    // Ensure the structure exists
    if (!isset($tfa_user_settings['data'])) {
      $tfa_user_settings['data'] = [];
    }
    if (!isset($tfa_user_settings['data']['plugins'])) {
      $tfa_user_settings['data']['plugins'] = [];
    }
    
    // Add tfa_email_support to existing plugins
    $tfa_user_settings['data']['plugins']['tfa_email_support'] = 'tfa_email_support';
    $tfa_user_settings['saved'] = $current_time;
    $tfa_user_settings['status'] = 1;
  } else {
    // Build new tfa_user_settings structure
    $tfa_user_settings = [
      'saved' => $current_time,
      'status' => 1,
      'data' => [
        'plugins' => [
          'tfa_email_support' => 'tfa_email_support',
        ],
        'sms' => 0,
      ],
      'validation_skipped' => 0,
    ];
  }

  $this->setUsersDataValue($this->uid, 'tfa', 'tfa_user_settings', $tfa_user_settings);

  // Config system - also check for existing config
  $config = \Drupal::configFactory()->getEditable("tfa.settings.user.{$this->uid}");
  
  // Get existing validation plugins or start with empty array
  $existing_validation_plugins = $config->get('validation_plugins') ?: [];
  
  // Add tfa_email_support if not already present
  if (!in_array('tfa_email_support', $existing_validation_plugins)) {
    $existing_validation_plugins[] = 'tfa_email_support';
  }
  
  $plugin_data = [
    'status' => 1,
    'created' => $current_time,
    'saved' => $current_time,
    'email' => $email,
  ];

  $config->set('validation_plugins', $existing_validation_plugins);
  $config->set('status', TRUE);
  $config->set('saved', $current_time);
  $config->set('data.tfa_email_support', $plugin_data);

  if ($config->isNew()) {
    $config->set('required', TRUE);
    $config->set('allowed_validation_methods', $existing_validation_plugins);
  }

  $config->save();

  $this->messenger->addStatus($this->t(
    'TFA via Email enabled for user @uid at @mail. users_data updated.',
    ['@uid' => $this->uid, '@mail' => $email]
  ));
}


protected function disableEmailTfa() {
  $connection = \Drupal::database();

  \Drupal::logger('tfa_email_support')->notice('Disabling TFA Email for user @uid', ['@uid' => $this->uid]);

  // Get all TFA plugin records for this user (exclude general settings and email).
  $tfa_records = $connection->select('users_data', 'ud')
    ->fields('ud', ['name', 'value'])
    ->condition('uid', $this->uid)
    ->condition('module', 'tfa')
    ->condition('name', 'tfa_email_support', '<>') 
    ->condition('name', 'tfa_user_settings', '<>') 
    ->execute()
    ->fetchAllKeyed();

  // Simply check if there are any other records
  $has_other_active_methods = !empty($tfa_records);

  // Always remove tfa_email_support from users_data
  $this->deleteUsersDataRecord($this->uid, 'tfa', 'tfa_email_support');

  if (!$has_other_active_methods) {
    // No other methods, remove user settings and config
    $this->deleteUsersDataRecord($this->uid, 'tfa', 'tfa_user_settings');

    $editable_config = \Drupal::configFactory()->getEditable("tfa.settings.user.{$this->uid}");
    if (!$editable_config->isNew()) {
      $editable_config->delete();
    }

    $this->messenger->addStatus($this->t(
      'TFA via Email disabled for user @uid. All TFA settings removed since no other TFA methods are active.',
      ['@uid' => $this->uid]
    ));
  }
  else {
    // Other methods exist, update tfa_user_settings to remove tfa_email_support from plugins
    $existing_tfa_user_settings = $connection->select('users_data', 'ud')
      ->fields('ud', ['value'])
      ->condition('uid', $this->uid)
      ->condition('module', 'tfa')
      ->condition('name', 'tfa_user_settings')
      ->execute()
      ->fetchField();

    if ($existing_tfa_user_settings) {
      $tfa_user_settings = unserialize($existing_tfa_user_settings, ['allowed_classes' => false]);
      
      // Remove tfa_email_support from plugins array if it exists
      if (isset($tfa_user_settings['data']['plugins']['tfa_email_support'])) {
        unset($tfa_user_settings['data']['plugins']['tfa_email_support']);
        
        // Update the saved timestamp
        $tfa_user_settings['saved'] = (int) \Drupal::time()->getRequestTime();
        
        // Save the updated tfa_user_settings
        $this->setUsersDataValue($this->uid, 'tfa', 'tfa_user_settings', $tfa_user_settings);
      }
    }

    // Update config to remove only tfa_email_support
    $editable_config = \Drupal::configFactory()->getEditable("tfa.settings.user.{$this->uid}");

    if (!$editable_config->isNew()) {
      $validation_plugins = $editable_config->get('validation_plugins') ?: [];

      // Remove tfa_email_support from validation_plugins
      $updated_plugins = array_values(array_diff($validation_plugins, ['tfa_email_support']));
      $editable_config->set('validation_plugins', $updated_plugins);

      // Remove email plugin data
      $editable_config->clear('data.tfa_email_support');

      // If no plugins left, disable TFA entirely
      if (empty($updated_plugins)) {
        $editable_config->set('status', FALSE);
      }

      $editable_config->save();
    }

    $this->messenger->addStatus($this->t(
      'TFA via Email disabled for user @uid. Other TFA methods remain active.',
      ['@uid' => $this->uid]
    ));
  }
}


  /**
   * Check if there are other active TFA methods in users_data.
   */
  protected function checkForOtherActiveTfaMethods($validation_plugins) {
    if (empty($validation_plugins)) {
      return FALSE;
    }

    $connection = \Drupal::database();
    
    foreach ($validation_plugins as $plugin) {
      if ($plugin === 'tfa_email_support') {
        continue;
      }
      
      // Check if this plugin has active data in users_data
      $plugin_data = $connection->select('users_data', 'ud')
        ->fields('ud', ['value'])
        ->condition('uid', $this->uid)
        ->condition('module', 'tfa')
        ->condition('name', $plugin)
        ->execute()
        ->fetchField();
        
      if ($plugin_data) {
        $data = unserialize($plugin_data, ['allowed_classes' => false]);
        if (is_array($data) && !empty($data['status'])) {
          return TRUE;
        } elseif ($data == 1) {
          return TRUE;
        }
      }
    }
    
    return FALSE;
  }

protected function setUsersDataValue($uid, $module, $name, $value) {
  $connection = \Drupal::database();
  $serialized_value = serialize($value); 
  $connection->merge('users_data')
    ->key(['uid' => $uid, 'module' => $module, 'name' => $name])
    ->fields(['value' => $serialized_value, 'serialized' => 1])
    ->execute();
}


  protected function deleteUsersDataRecord($uid, $module, $name) {
    $connection = \Drupal::database();
    $connection->delete('users_data')
      ->condition('uid', $uid)
      ->condition('module', $module)
      ->condition('name', $name)
      ->execute();
  }

  protected function clearTfaCaches() {
    if (\Drupal::hasService('plugin.manager.tfa.validation')) {
      \Drupal::service('plugin.manager.tfa.validation')->clearCachedDefinitions();
    }
    if (\Drupal::hasService('plugin.manager.tfa.setup')) {
      \Drupal::service('plugin.manager.tfa.setup')->clearCachedDefinitions();
    }
    if (\Drupal::hasService('cache_tags.invalidator')) {
      \Drupal::service('cache_tags.invalidator')->invalidateTags([
        "user:{$this->uid}",
        "config:tfa.settings.user.{$this->uid}",
        'tfa_validation_plugins',
      ]);
    }
    \Drupal::cache('config')->deleteAll();
  }
}