<?php

namespace Drupal\tfa_email_support\Plugin\TfaValidation;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\tfa\Plugin\TfaValidationInterface;
use Drupal\tfa\Plugin\TfaSendInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Email OTP TFA validation plugin.
 *
 * @TfaValidation(
 *   id = "tfa_email_support",
 *   label = @Translation("Email OTP"),
 *   description = @Translation("Validates using a one-time password sent via email.")
 * )
 */
class Email extends PluginBase implements TfaValidationInterface, TfaSendInterface, ContainerFactoryPluginInterface {

  protected $context;
  protected $state;
  protected $mailManager;
  protected $languageManager;
  protected $configFactory;
  protected $logger;
  protected $moduleHandler;
  protected $messenger;
  protected $currentUser;
  protected $entityTypeManager;

  const COOLDOWN_PERIOD = 120;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    StateInterface $state,
    MailManagerInterface $mail_manager,
    LanguageManagerInterface $language_manager,
    ConfigFactoryInterface $config_factory,
    LoggerInterface $logger,
    ModuleHandlerInterface $module_handler,
    MessengerInterface $messenger,
    AccountProxyInterface $current_user,
    EntityTypeManagerInterface $entity_type_manager,
    array $context = []
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->state = $state;
    $this->mailManager = $mail_manager;
    $this->languageManager = $language_manager;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
    $this->moduleHandler = $module_handler;
    $this->messenger = $messenger;
    $this->currentUser = $current_user;
    $this->entityTypeManager = $entity_type_manager;
    $this->context = $context;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $context = $configuration;

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('state'),
      $container->get('plugin.manager.mail'),
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('logger.factory')->get('tfa_email_support'),
      $container->get('module_handler'),
      $container->get('messenger'),
      $container->get('current_user'),
      $container->get('entity_type.manager'),
      $context
    );
  }

  public function getForm(array $form, FormStateInterface $form_state) {
    $uid = $this->context['uid'] ?? $this->currentUser->id();
    // $this->logger->notice('getForm() called. UID: @uid', ['@uid' => $uid]);

    if ($uid) {
      $stored_otp_data = $this->state->get('tfa_email_support_otp_' . $uid);
      $send_new_otp = empty($stored_otp_data) || (is_array($stored_otp_data) && time() > $stored_otp_data['expiry']);
      if ($send_new_otp) {
        // $this->logger->notice('Sending new OTP for UID: @uid', ['@uid' => $uid]);
        $this->begin();
      }
    }

    $email = $this->getTfaEmail($uid);
    $masked_email = $this->maskEmail($email);

    $form['message'] = [
      '#markup' => '<div class="messages messages--status">' .
        $this->t('A verification code has been sent to @email.', ['@email' => $masked_email]) .
        '</div>',
      '#weight' => -10,
    ];

    $form['otp'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter the verification code sent to your email'),
      '#required' => TRUE,
      '#maxlength' => 6,
      '#size' => 10,
      '#attributes' => [
        'autocomplete' => 'off',
        'placeholder' => $this->t('Enter 6-digit code'),
      ],
    ];

    $form['button_container'] = [
      '#type' => 'container',
      '#weight' => 10,
      '#attributes' => ['class' => ['form-actions', 'tfa-button-container']],
    ];

    $form['button_container']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Verify'),
      '#attributes' => ['class' => ['button', 'button--primary', 'form-submit']],
    ];

    $form['button_container']['resend_container'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'resend-container', 'style' => 'display: inline-block;'],
    ];

    $this->buildResendSection($form['button_container']['resend_container'], $uid);
    $form['#attached']['library'][] = 'tfa_email_support/resend_timer';

    return $form;
  }

  private function buildResendSection(array &$container, $uid) {
    $last_sent_data = $this->state->get('tfa_email_support_last_sent_' . $uid);
    $remaining_time = $last_sent_data && is_array($last_sent_data) ? max(0, self::COOLDOWN_PERIOD - (time() - $last_sent_data['timestamp'])) : 0;

    if ($remaining_time === 0) {
      $container['resend'] = [
        '#type' => 'submit',
        '#value' => $this->t('Resend Code'),
        '#submit' => ['::resendCode'],
        '#limit_validation_errors' => [],
        '#ajax' => [
          'callback' => '::ajaxResendCallback',
          'wrapper' => 'resend-container',
          'method' => 'replace',
          'effect' => 'fade',
        ],
      ];
    }
    else {
      $time_display = $this->t('@sec seconds', ['@sec' => $remaining_time]);
      $container['resend_message'] = [
        '#markup' => '<div class="resend-timer" data-remaining="' . $remaining_time . '">' .
          $this->t('You can request a new code in <span class="countdown">@time</span>', ['@time' => $time_display]) .
          '</div>',
      ];
    }
  }

  public function ajaxResendCallback(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#resend-container', $form['button_container']['resend_container']));
    return $response;
  }

  public function resendCode(array &$form, FormStateInterface $form_state) {
    $this->begin();
    $this->messenger->addStatus($this->t('A new verification code has been sent.'));
  }

  public function validateForm(array $form, FormStateInterface $form_state) {
    // $this->logger->notice('validateForm() called');

    $otp = trim($form_state->getValue('otp'));
    $uid = $this->context['uid'] ?? $this->currentUser->id();
    // $this->logger->notice('Submitted OTP: @otp for UID: @uid', ['@otp' => $otp, '@uid' => $uid]);

    $data = $this->state->get('tfa_email_support_otp_' . $uid);
    if (empty($data) || $otp !== ($data['otp'] ?? '')) {
      $form_state->setErrorByName('otp', $this->t('Invalid or expired code.'));
      return FALSE;
    }
    return TRUE;
  }

  public function submitForm(array $form, FormStateInterface $form_state) {
    // $this->logger->notice('submitForm() called');
    $uid = $this->context['uid'] ?? $this->currentUser->id();
    $this->state->delete('tfa_email_support_otp_' . $uid);
    $this->state->delete('tfa_email_support_last_sent_' . $uid);
  }

  public function isComplete() {
    return TRUE;
  }

  public function begin() {
    $uid = $this->context['uid'] ?? $this->currentUser->id();
    // $this->logger->notice('begin() called. UID: @uid', ['@uid' => $uid]);

    if (!$uid) {
      $this->logger->warning('begin(): UID is missing or zero.');
      return FALSE;
    }

    $email = $this->getTfaEmail($uid);
    if (empty($email)) {
      $this->logger->warning('begin(): No TFA email configured for UID: @uid.', ['@uid' => $uid]);
      return FALSE;
    }

    $otp = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);
    $expiry = time() + self::COOLDOWN_PERIOD;

    $this->state->set('tfa_email_support_otp_' . $uid, ['otp' => $otp, 'expiry' => $expiry]);
    $this->state->set('tfa_email_support_last_sent_' . $uid, ['timestamp' => time()]);
    // $this->logger->notice('Generated OTP: @otp for email: @email', ['@otp' => $otp, '@email' => $email]);

    $module = 'tfa_email_support';
    $key = 'send_otp';
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    $site_mail = $this->configFactory->get('system.site')->get('mail');
    $params = ['otp' => $otp, 'uid' => $uid];

    $result = $this->mailManager->mail($module, $key, $email, $langcode, $params, $site_mail, TRUE);

    if ($result['result'] ?? FALSE) {
      $this->logger->info('OTP email successfully sent to @email for UID @uid.', ['@email' => $email, '@uid' => $uid]);
      return TRUE;
    }
    else {
      $this->logger->error('Failed to send OTP email to @email for UID @uid.', ['@email' => $email, '@uid' => $uid]);
      return FALSE;
    }
  }

  protected function getTfaEmail($uid) {
    if (!$uid) {
      $uid = $this->currentUser->id();
    }
    $config = $this->configFactory->get("tfa.settings.user.$uid");
    $tfa_email_support = $config->get('data.tfa_email_support.email');

    if (!empty($tfa_email_support)) {
      return $tfa_email_support;
    }

    $user_storage = $this->entityTypeManager->getStorage('user');
    $user = $user_storage->load($uid);
    return $user ? $user->getEmail() : NULL;
  }

  protected function maskEmail($email) {
    if (empty($email)) {
      return '';
    }
    $parts = explode('@', $email);
    if (count($parts) !== 2) {
      return $email;
    }
    $username = $parts[0];
    $domain = $parts[1];
    $username_length = strlen($username);

    if ($username_length <= 3) {
      $masked_username = substr($username, 0, 1) . '*' . substr($username, -1);
    }
    else {
      $masked_username = substr($username, 0, 2) . str_repeat('*', $username_length - 3) . substr($username, -1);
    }

    return $masked_username . '@' . $domain;
  }

  public function getConfiguration() {
    return $this->configuration;
  }

  public function setConfiguration(array $configuration) {
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  public function defaultConfiguration() {
    return [];
  }

 /**
 * Fixed ready(): Check both config and users_data for plugin status.
 */
public function ready() {
  \Drupal::logger('tfa_email_support')->notice('READY() called.');
  $uid = $this->context['uid'] ?? $this->currentUser->id();
  // $this->logger->notice('READY() called. UID: @uid', ['@uid' => $uid]);

  if (!$uid) {
    $this->logger->warning('READY(): UID is missing or zero.');
    return FALSE;
  }
  $config = $this->configFactory->get("tfa.settings.user.{$uid}");
  $validation_plugins = $config->get('validation_plugins') ?: [];
  $is_enabled_in_config = in_array('tfa_email_support', $validation_plugins);
  
  $connection = \Drupal::database();
  $tfa_email_support = $connection->select('users_data', 'ud')
    ->fields('ud', ['value'])
    ->condition('uid', $uid)
    ->condition('module', 'tfa')
    ->condition('name', 'tfa_email_support')
    ->execute()
    ->fetchField();

  $is_enabled_in_users_data = FALSE;
  if ($tfa_email_support) {
    $tfa_email_support_data = unserialize($tfa_email_support, ['allowed_classes' => false]);

    if (is_array($tfa_email_support_data)) {
      $is_enabled_in_users_data = !empty($tfa_email_support_data['status']);
    } else {
      $is_enabled_in_users_data = ($tfa_email_support_data == 1);
    }
  }

  if (!$is_enabled_in_config && !$is_enabled_in_users_data) {
    // $this->logger->notice('READY(): tfa_email_support not enabled for UID: @uid', ['@uid' => $uid]);
    return FALSE;
  }

  $user_storage = $this->entityTypeManager->getStorage('user');
  $user = $user_storage->load($uid);

  if (!$user) {
    $this->logger->warning('READY(): Could not load user entity for UID: @uid', ['@uid' => $uid]);
    return FALSE;
  }

  $configured_email = $config->get('data.tfa_email_support.email');
  $email = $configured_email ?: $user->getEmail();

  if (empty($email)) {
    $this->logger->warning('READY(): No email configured for UID: @uid', ['@uid' => $uid]);
    return FALSE;
  }

  // $this->logger->notice('READY(): TFA Email ready for UID: @uid, Email: @email', [
  //   '@uid' => $uid,
  //   '@email' => $email,
  // ]);

  // Return TRUE for TFA system compatibility
  return TRUE;
}

  public function getDescription() {
    $uid = $this->context['uid'] ?? $this->currentUser->id();

    if (!$uid) {
      return $this->t('Email-based two-factor authentication.');
    }

    $ready = $this->ready();
    if (is_array($ready) && $ready['status']) {
      $masked_email = $this->maskEmail($ready['email']);
      return $this->t('Email TFA configured for: @email', ['@email' => $masked_email]);
    }

    return $this->t('Setup email-based two-factor authentication.');
  }

  public function getLabel() {
    $uid = $this->context['uid'] ?? $this->currentUser->id();

    if (!$uid) {
      return $this->t('Email OTP');
    }

    $ready = $this->ready();
    if (is_array($ready) && $ready['status']) {
      $masked_email = $this->maskEmail($ready['email']);
      return $this->t('Email OTP (@email)', ['@email' => $masked_email]);
    }

    return $this->t('Email OTP');
  }

  public function getCodeLength() {
    return 6;
  }

  public function sendCode() {
    return $this->begin();
  }
}
