(function ($, Drupal) {
    'use strict';

    let countdownInterval;

    /**
     * Start countdown timer.
     */
    function startCountdown(remainingTime) {
        if (countdownInterval) {
            clearInterval(countdownInterval);
        }

        const countdownElement = $('.resend-timer .countdown');
        if (countdownElement.length === 0) {
            return;
        }

        let timeLeft = remainingTime;

        countdownInterval = setInterval(function () {
            if (timeLeft <= 0) {
                clearInterval(countdownInterval);
                showResendButton();
                return;
            }

            const minutes = Math.floor(timeLeft / 60);
            const seconds = timeLeft % 60;
            const timeDisplay = minutes > 0 ?
                minutes + ' min ' + seconds + ' sec' :
                seconds + ' seconds';

            countdownElement.text(timeDisplay);
            timeLeft--;
        }, 1000);
    }

    /**
     * Replace timer with resend button.
     */
    function showResendButton() {
        const resendContainer = $('#resend-container');
        if (resendContainer.length > 0) {
            const resendButtonHtml = '<button type="submit" name="resend" value="Resend Code" class="button form-submit" style="margin-left: 10px;" data-drupal-selector="edit-resend">Resend Code</button>';

            resendContainer.html(resendButtonHtml);
            resendContainer.find('button[name="resend"]').on('click', function (e) {
                e.preventDefault();
                const $button = $(this);
                $button.prop('disabled', true).text('Send');
                const originalForm = $(this).closest('form');
                const formData = new FormData(originalForm[0]);
                formData.set('op', 'Resend Code');
                formData.set('_triggering_element_name', 'resend');
                formData.set('_triggering_element_value', 'Resend Code');
                formData.set('_drupal_ajax', '1');
                $.ajax({
                    url: originalForm.attr('action') || window.location.pathname,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    headers: {
                        'X-Requested-With': 'XMLHttpRequest'
                    },
                    success: function (response) {
                        if (response && response.length > 0) {
                            for (let i = 0; i < response.length; i++) {
                                if (response[i].command === 'replace' && response[i].selector === '#resend-container') {
                                    $(response[i].selector).replaceWith(response[i].data);
                                    Drupal.attachBehaviors($('#resend-container')[0]);
                                } else if (response[i].command === 'invoke' && response[i].method === 'startCountdown') {
                                    startCountdown(response[i].arguments[0]);
                                }
                            }
                        }
                    },
                    error: function () {
                        $button.prop('disabled', false).text('Resend Code');
                        $('<div class="messages messages--error">Failed to send verification code. Please try again.</div>')
                            .insertBefore(originalForm);
                    }
                });
            });
        }
    }

    /**
     * Initialize countdown on page load if timer exists.
     */
    Drupal.behaviors.tfaEmailResendTimer = {
        attach: function (context, settings) {
            const timerElement = $('.resend-timer[data-remaining]', context);
            if (timerElement.length > 0) {
                const remainingTime = parseInt(timerElement.attr('data-remaining'), 10);
                if (remainingTime > 0) {
                    startCountdown(remainingTime);
                }
            }
        }
    };

    window.startCountdown = startCountdown;

})(jQuery, Drupal);