<?php

namespace Drupal\tfa_email_support\Plugin\TfaSetup;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\tfa\Plugin\TfaSetupInterface;
use Drupal\tfa\Plugin\TfaValidationInterface;
use Drupal\user\Entity\User;
use Drupal\tfa_email_support\Form\EnableTfaEmailForm;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Routing\RouteMatchInterface;

/**
 * Email TFA setup plugin.
 *
 * @TfaSetup(
 *   id = "tfa_email_support_setup",
 *   label = @Translation("Email Setup"),
 *   description = @Translation("Setup email-based two-factor authentication.")
 * )
 */
class EmailSetup implements TfaSetupInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  protected $validationPlugin;
  protected $user;
  protected $setupStep = 'email_input';
  protected $mailManager;
  protected $state;
  protected $messenger;
  protected $configFactory;
  protected $languageManager;
  protected $logger;
  protected $formBuilder;

  /**
   * Constructs a new EmailSetup object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    User $user,
    TfaValidationInterface $validation_plugin,
    MailManagerInterface $mail_manager,
    StateInterface $state,
    MessengerInterface $messenger,
    ConfigFactoryInterface $config_factory,
    LanguageManagerInterface $language_manager,
    LoggerInterface $logger,
    FormBuilderInterface $form_builder
  ) {
    $this->user = $user;
    $this->validationPlugin = $validation_plugin;
    $this->mailManager = $mail_manager;
    $this->state = $state;
    $this->messenger = $messenger;
    $this->configFactory = $config_factory;
    $this->languageManager = $language_manager;
    $this->logger = $logger;
    $this->formBuilder = $form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var RouteMatchInterface $route_match */
    $route_match = $container->get('current_route_match');
    $user_param = $route_match->getParameter('user');

    if ($user_param instanceof User) {
      $user_entity = $user_param;
    }
    elseif (is_numeric($user_param)) {
      $user_entity = User::load($user_param);
    }
    else {
      $current_user = $container->get('current_user');
      $user_entity = User::load($current_user->id());
    }

    $uid = $user_entity ? $user_entity->id() : NULL;

    $validation_plugin_manager = $container->get('plugin.manager.tfa.validation');
    $validation_plugin = $validation_plugin_manager->createInstance('tfa_email_support', ['uid' => $uid]);

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $user_entity,
      $validation_plugin,
      $container->get('plugin.manager.mail'),
      $container->get('state'),
      $container->get('messenger'),
      $container->get('config.factory'),
      $container->get('language_manager'),
      $container->get('logger.factory')->get('tfa_email_support'),
      $container->get('form_builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupForm(array $form, FormStateInterface $form_state) {
    $setup_step = $form_state->get('setup_step') ?: $this->setupStep;

    // Assign validation and submit handlers once.
    $form['#validate'][] = [$this, 'validateSetupForm'];
    $form['#submit'][] = [$this, 'submitSetupForm'];

    switch ($setup_step) {
      case 'email_input':
        $sub_form = $this->getEmailInputForm([], $form_state);
        break;

      case 'email_verification':
        $sub_form = $this->getEmailVerificationForm([], $form_state);
        break;

      default:
        $sub_form = [];
        break;
    }

    // Merge sub-form elements with the main form.
    return $form + $sub_form;
  }

  /**
   * Email input form step.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @return array
   *   The form array.
   */
  protected function getEmailInputForm(array $form, FormStateInterface $form_state) {
    $current_email = $this->user->getEmail();

    $form['email_setup'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Email Two-Factor Authentication Setup'),
      '#description' => $this->t('Enter the email address where you want to receive verification codes for two-factor authentication.'),
    ];

    if (!empty($current_email)) {
      $form['email_setup']['current_email_info'] = [
        '#markup' => '<p>' . $this->t('Your current account email is: <strong>@email</strong>', ['@email' => $current_email]) . '</p>',
      ];

      $form['email_setup']['use_current_email'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Use my current account email for TFA'),
        '#default_value' => TRUE,
      ];
    }

    $form['email_setup']['custom_email'] = [
      '#type' => 'email',
      '#title' => $this->t('Email address for TFA codes'),
      '#description' => $this->t('Enter the email address where verification codes should be sent.'),
      '#required' => empty($current_email),
      '#states' => [
        'visible' => [
          ':input[name="use_current_email"]' => ['checked' => FALSE],
        ],
        'required' => [
          ':input[name="use_current_email"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['email_setup']['actions'] = [
      '#type' => 'actions',
    ];

    $form['email_setup']['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send Verification Code'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * Email verification form step.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *
   * @return array
   *   The form array.
   */
  protected function getEmailVerificationForm(array $form, FormStateInterface $form_state) {
    $email = $form_state->get('tfa_email_support');

    $form['email_verification'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Verify Your Email'),
      '#description' => $this->t('A verification code has been sent to <strong>@email</strong>. Enter the code below to complete setup.', ['@email' => $email]),
    ];

    $form['email_verification']['verification_code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Verification Code'),
      '#description' => $this->t('Enter the 6-digit code sent to your email.'),
      '#required' => TRUE,
      '#maxlength' => 6,
      '#size' => 10,
      '#attributes' => [
        'autocomplete' => 'off',
        'placeholder' => $this->t('Enter 6-digit code'),
      ],
    ];

    $form['email_verification']['actions'] = [
      '#type' => 'actions',
    ];

    $form['email_verification']['actions']['verify'] = [
      '#type' => 'submit',
      '#value' => $this->t('Verify and Complete Setup'),
      '#button_type' => 'primary',
    ];

    $form['email_verification']['actions']['resend'] = [
      '#type' => 'submit',
      '#value' => $this->t('Resend Code'),
      '#submit' => [[$this, 'resendVerificationCode']],
      '#limit_validation_errors' => [],
    ];

    $form['email_verification']['actions']['back'] = [
      '#type' => 'submit',
      '#value' => $this->t('Change Email Address'),
      '#submit' => [[$this, 'goBackToEmailInput']],
      '#limit_validation_errors' => [],
    ];

    return $form;
  }

  /**
   * Validate setup form dispatcher.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function validateSetupForm(array $form, FormStateInterface $form_state): void {
    $setup_step = $form_state->get('setup_step') ?: $this->setupStep;

    if ($setup_step === 'email_input') {
      $this->validateEmailInput($form, $form_state);
    }
    elseif ($setup_step === 'email_verification') {
      $this->validateEmailVerification($form, $form_state);
    }
  }

  /**
   * Submit setup form dispatcher.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function submitSetupForm(array $form, FormStateInterface $form_state): void {
    $setup_step = $form_state->get('setup_step') ?: $this->setupStep;

    if ($setup_step === 'email_input') {
      $this->submitEmailInput($form, $form_state);
    }
    elseif ($setup_step === 'email_verification') {
      $this->submitEmailVerification($form, $form_state);
    }
  }

  /**
   * Validate email input step.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function validateEmailInput(array $form, FormStateInterface $form_state): void {
    $use_current_email = $form_state->getValue('use_current_email');
    $custom_email = $form_state->getValue('custom_email');
    $current_email = $this->user->getEmail();

    if (!$use_current_email && empty($custom_email)) {
      $form_state->setErrorByName('custom_email', $this->t('Please enter an email address.'));
      return;
    }

    $email_to_use = $use_current_email ? $current_email : $custom_email;

    if (empty($email_to_use)) {
      $form_state->setErrorByName('custom_email', $this->t('Email address is required.'));
      return;
    }

    if (!filter_var($email_to_use, FILTER_VALIDATE_EMAIL)) {
      $field_name = $use_current_email ? 'use_current_email' : 'custom_email';
      $form_state->setErrorByName($field_name, $this->t('Please enter a valid email address.'));
    }
  }

  /**
   * Submit email input step.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function submitEmailInput(array $form, FormStateInterface $form_state): void {
    $use_current_email = $form_state->getValue('use_current_email');
    $custom_email = $form_state->getValue('custom_email');
    $current_email = $this->user->getEmail();

    $email_to_use = $use_current_email ? $current_email : $custom_email;

    $form_state->set('tfa_email_support', $email_to_use);
    $form_state->set('setup_step', 'email_verification');

    $this->sendVerificationCode($email_to_use);

    $form_state->setRebuild(TRUE);

    $this->messenger->addStatus($this->t('A verification code has been sent to @email', ['@email' => $email_to_use]));
  }

  /**
   * Validate email verification step.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function validateEmailVerification(array $form, FormStateInterface $form_state): void {
    $submitted_code = trim($form_state->getValue('verification_code'));
    $uid = $this->user->id();

    if (empty($submitted_code)) {
      $form_state->setErrorByName('verification_code', $this->t('Please enter the verification code.'));
      return;
    }

    $stored_data = $this->state->get('tfa_email_support_otp_' . $uid);

    if (empty($stored_data) || $submitted_code !== ($stored_data['otp'] ?? '')) {
      $form_state->setErrorByName('verification_code', $this->t('Invalid or expired verification code. Please try again.'));
      return;
    }

    if (time() > ($stored_data['expiry'] ?? 0)) {
      $form_state->setErrorByName('verification_code', $this->t('Verification code has expired. Please request a new code.'));
      return;
    }
  }

  /**
   * Submit email verification step.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  protected function submitEmailVerification(array $form, FormStateInterface $form_state): void {
    $email = $form_state->get('tfa_email_support');
    $uid = $this->user->id();

    $this->saveTfaEmailConfiguration($email);

    $this->state->delete('tfa_email_support_otp_' . $uid);
    $this->state->delete('tfa_email_support_last_sent_' . $uid);

    $form_state->set('setup_complete', TRUE);

    $this->messenger->addStatus($this->t('Email two-factor authentication has been successfully set up for @email', ['@email' => $email]));
  }

  /**
   * Resend verification code handler.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function resendVerificationCode(array $form, FormStateInterface $form_state): void {
    $email = $form_state->get('tfa_email_support');
    $this->sendVerificationCode($email);
    $form_state->setRebuild(TRUE);
    $this->messenger->addStatus($this->t('A new verification code has been sent to @email', ['@email' => $email]));
  }

  /**
   * Go back to email input step handler.
   *
   * @param array $form
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function goBackToEmailInput(array $form, FormStateInterface $form_state): void {
    $form_state->set('setup_step', 'email_input');
    $form_state->setRebuild(TRUE);
  }

  /**
   * Send verification code to email.
   *
   * @param string $email
   *   Email address.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  protected function sendVerificationCode(string $email): bool {
    $uid = $this->user->id();

    try {
      $otp = str_pad(random_int(0, 999999), 6, '0', STR_PAD_LEFT);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to generate secure OTP for user @uid: @message', ['@uid' => $uid, '@message' => $e->getMessage()]);
      return FALSE;
    }

    $expiry = time() + 300;

    $this->state->set('tfa_email_support_otp_' . $uid, [
      'otp' => $otp,
      'expiry' => $expiry,
      'email' => $email,
    ]);

    $site_mail = $this->configFactory->get('system.site')->get('mail');

    $params = [
      'otp' => $otp,
      'setup_mode' => TRUE,
    ];

    $result = $this->mailManager->mail('tfa_email_support', 'send_setup_otp', $email,
      $this->languageManager->getCurrentLanguage()->getId(), $params, $site_mail, TRUE);

    if (!($result['result'] ?? FALSE)) {
      $this->logger->error('Failed to send verification code email to @email for user @uid.', ['@email' => $email, '@uid' => $uid]);
      return FALSE;
    }

    $this->logger->info('Setup verification code sent to @email for user @uid', [
      '@email' => $email,
      '@uid' => $uid,
    ]);

    return TRUE;
  }

  /**
   * Save TFA email configuration.
   *
   * @param string $email
   *   Email address.
   */
  protected function saveTfaEmailConfiguration(string $email): void {
    $uid = $this->user->id();

    $config = $this->configFactory->getEditable("tfa.settings.user.$uid");

    $plugins = $config->get('validation_plugins') ?? [];
    if (!in_array('tfa_email_support', $plugins)) {
      $plugins[] = 'tfa_email_support';
    }

    $config->set('validation_plugins', $plugins);
    $config->set('data.tfa_email_support.email', $email);
    $config->save();

    $this->logger->info('TFA email configuration saved for user @uid with email @email', [
      '@uid' => $uid,
      '@email' => $email,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getOverview(array $params) {
    $uid = $this->user->id();
    $config = $this->configFactory->get("tfa.settings.user.$uid");
    $email = $config->get('data.tfa_email_support.email') ?: $this->user->getEmail();

    return [
      'heading' => [
        '#type' => 'html_tag',
        '#tag' => 'h3',
        '#value' => $this->t('Email Two-Factor Authentication'),
      ],
      'description' => [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t('Verification codes will be sent to: @email', ['@email' => $email]),
      ],
      'form' => $this->formBuilder->getForm(EnableTfaEmailForm::class, $uid),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getHelpLinks() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupMessages() {
    return [
      'saved' => $this->t('Email two-factor authentication setup complete.'),
      'skipped' => $this->t('Email two-factor authentication setup skipped.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getErrorMessages() {
    return [
      'invalid_code' => $this->t('Invalid or expired verification code. Please try again.'),
      'expired_code' => $this->t('Verification code has expired. Please request a new code.'),
      'missing_code' => $this->t('Please enter the verification code.'),
      'invalid_email' => $this->t('Please enter a valid email address.'),
      'email_required' => $this->t('Email address is required.'),
      'send_failed' => $this->t('Failed to send verification code. Please try again.'),
    ];
  }

}
