<?php

namespace Drupal\tfa_headless\EventSubscriber;

use Drupal\Component\Utility\Crypt;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\Routing\RequestContext;

/**
 * The OauthReponseSubscriber class.
 */
class OauthResponseSubscriber implements EventSubscriberInterface {

  /**
   * Constructs a new OauthResponseSubscriber object.
   */
  public function __construct(
    protected RequestContext $requestContext,
    protected Session $session,
    protected LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::RESPONSE => ['onKernelResponse', -10],
    ];
  }

  /**
   * Alter info returned by the /oauth/token endpoint.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onKernelResponse(ResponseEvent $event) {
    $request = $event->getRequest();
    $path = $this->requestContext->getPathInfo();
    // Only override if session parameter is given.
    if (str_contains($path, '/oauth/token') && $request->request->get('session') == TRUE && $request->isMethod('POST')) {
      try {
        $response = $event->getResponse();
        $content = json_decode($response->getContent(), TRUE);

        // Log the decoded content for debugging.
        $this->logger->info('OAuth response content: @content', [
          '@content' => print_r($content, TRUE),
        ]);

        if (json_last_error() === JSON_ERROR_NONE && is_array($content)) {
          $this->session->setId(Crypt::randomBytesBase64());
          if (!empty($content['access_token'])) {
            $this->session->set('access_token', json_encode($content));
            $content = ['session' => $this->session->getId()];
          }
          else {
            // Log when access_token is missing.
            $this->logger->warning('access_token missing from OAuth response. Available keys: @keys', [
              '@keys' => implode(', ', array_keys($content)),
            ]);
          }

          $response->setContent(json_encode($content));
          $response->headers->set('Content-Type', 'application/json');
        }
        else {
          $this->logger->error('Failed to decode OAuth response JSON. Error: @error', [
            '@error' => json_last_error_msg(),
          ]);
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Exception in OAuth response subscriber: @message', [
          '@message' => $e->getMessage(),
        ]);
      }
    }
  }

}
