<?php

namespace Drupal\tfa_headless\Plugin\rest\resource;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Drupal\tfa\TfaLoginContextTrait;
use Drupal\tfa_headless\Service\TfaHeadlessService;
use Drupal\user\UserDataInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Get the status of the current user's 2fa state.
 *
 * @RestResource(
 *   id = "tfa_headless_status",
 *   label = @Translation("TFA Headless Status"),
 *   uri_paths = {
 *     "canonical" = "/api/totp/status"
 *   }
 * )
 */
class Status extends ResourceBase {
  use TfaLoginContextTrait;

  /**
   * The Headless TFA service.
   *
   * @var \Drupal\tfa_headless\Service\TfaHeadlessService
   */
  protected $tfaService;
  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new Status object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    UserDataInterface $user_data,
    RequestStack $request_stack,
    TfaHeadlessService $tfa_service,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);

    $this->userData = $user_data;
    $this->requestStack = $request_stack;
    $this->tfaService = $tfa_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('rest'),
      $container->get('user.data'),
      $container->get('request_stack'),
      $container->get('tfa_headless.service')
    );
  }

  /**
   * Responds to GET requests.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The HTTP response object.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\HttpException
   *   Throws exception expected.
   */
  public function get() {
    $request = $this->requestStack->getCurrentRequest();

    // Use session so the token in the frontend does
    // not have to be saved before TFA is completed.
    $session = $this->tfaService->getSession($request->query->get('session'));
    $user = $this->tfaService->getUser($session['access_token']);

    $tfaData = $this->tfaGetTfaData($user->id(), $this->userData);
    $enabled = isset($tfaData['status']) && $tfaData['status'];

    $response = new ResourceResponse(['status' => $enabled && !empty($tfaData['data']['plugins'])], 200);

    // Disable cache.
    $cache = new CacheableMetadata();
    $cache->setCacheMaxAge(0);
    $response->addCacheableDependency($cache);

    return $response;
  }

}
