<?php

namespace Drupal\tfa_headless\Plugin\rest\resource;

use Drupal\Component\Serialization\Json;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\tfa\TfaUserDataTrait;
use Drupal\tfa_headless\Service\TfaHeadlessService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Login the user with TFA.
 *
 * @RestResource(
 *   id = "tfa_headless_login",
 *   label = @Translation("TFA Headless Login"),
 *   uri_paths = {
 *     "create" = "/api/totp/login"
 *   }
 * )
 */
class Login extends ResourceBase {
  use TfaUserDataTrait;

  /**
   * The tfa config.
   *
   * @var mixed
   */
  protected $tfaConfig;
  /**
   * Encryption profile.
   *
   * @var \Drupal\encrypt\EncryptionProfileInterface
   */
  protected $encryptionProfile;
  /**
   * The Headless TFA service.
   *
   * @var \Drupal\tfa_headless\Service\TfaHeadlessService
   */
  protected $tfaService;

  /**
   * Constructs a new Login object.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    array $serializer_formats,
    LoggerInterface $logger,
    TfaHeadlessService $tfa_service,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);

    $this->tfaService = $tfa_service;
    $this->userData = $this->tfaService->getGlobalUserData();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('rest'),
      $container->get('tfa_headless.service'),
    );
  }

  /**
   * Responds to POST requests.
   *
   * @return \Drupal\rest\ResourceResponse
   *   The HTTP response object.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\HttpException
   *   Throws exception expected.
   */
  public function post(Request $request) {
    try {
      $payload = Json::decode($request->getContent());
    }
    catch (\InvalidArgumentException $exception) {
      return $this->tfaService->buildErrorResponse($this->t('Invalid JSON payload.'), 400);
    }

    if (!is_array($payload)) {
      return $this->tfaService->buildErrorResponse($this->t('Request payload must be a JSON object.'), 400);
    }

    $code = isset($payload['code']) ? (string) $payload['code'] : '';
    $sessionId = isset($payload['session']) ? (string) $payload['session'] : '';

    if ($code === '' || $sessionId === '') {
      return $this->tfaService->buildErrorResponse($this->t('Both code and session are required.'), 400);
    }

    $resolved = $this->tfaService->resolveSessionUser($sessionId);
    if ($resolved === NULL) {
      return $this->tfaService->buildErrorResponse($this->t('Session could not be loaded or has expired.'), 401);
    }

    $session = $resolved['session'];
    $user = $resolved['user'];

    $tfaData = $this->tfaGetTfaData($user->id(), $this->userData) ?? [];
    $enabled = !empty($tfaData['status']) && !empty($tfaData['data']['plugins']);
    if (!$enabled) {
      return $this->tfaService->buildErrorResponse($this->t('TFA is not enabled for this account.'), 401);
    }

    if (!$this->tfaService->validate($code)) {
      return $this->tfaService->buildErrorResponse($this->t('Invalid application code. Please try again.'), 401);
    }

    $this->tfaService->storeAcceptedCode($code);

    return $this->tfaService->buildResponse($session, 200);
  }

}
