<?php

namespace Drupal\tfa_headless\EventSubscriber;

use Drupal\Component\Serialization\Json;
use Drupal\Component\Utility\Crypt;
use Drupal\simple_oauth\Controller\Oauth2Token;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\Routing\RequestContext;

/**
 * The OauthReponseSubscriber class.
 */
class OauthResponseSubscriber implements EventSubscriberInterface {

  /**
   * Logger channel.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new OauthResponseSubscriber object.
   */
  public function __construct(
    protected RequestContext $requestContext,
    protected Session $session,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->logger = $logger_factory->get('tfa_headless');
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::RESPONSE => ['onKernelResponse', -10],
    ];
  }

  /**
   * Alter info returned by the /oauth/token endpoint.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onKernelResponse(ResponseEvent $event) {
    $request = $event->getRequest();
    $controller = (string) $request->attributes->get('_controller');
    $routeMatches = $this->isOauthTokenController($controller);
    $pathInfo = $request->getPathInfo();
    $contextPath = $this->requestContext->getPathInfo();
    $pathMatches = str_contains($pathInfo, '/oauth/token') || str_contains($contextPath, '/oauth/token');

    if ((!$routeMatches && !$pathMatches) || !$request->isMethod('POST') || !$this->sessionRequested($request->request->get('session'))) {
      return;
    }

    $response = $event->getResponse();
    if ($response->getStatusCode() !== 200) {
      return;
    }

    $contentType = $response->headers->get('Content-Type');
    if ($contentType !== NULL && stripos($contentType, 'application/json') === FALSE) {
      $this->logger->error('Unexpected Content-Type "@type" for OAuth response.', [
        '@type' => $contentType,
      ]);
      return;
    }

    try {
      $content = Json::decode((string) $response->getContent());
    }
    catch (\InvalidArgumentException $exception) {
      $this->logger->error('Failed to decode OAuth response JSON. Error: @error', [
        '@error' => $exception->getMessage(),
      ]);
      return;
    }
    catch (\Exception $exception) {
      $this->logger->error('Exception in OAuth response subscriber: @message', [
        '@message' => $exception->getMessage(),
      ]);
      return;
    }

    if (!is_array($content)) {
      $this->logger->error('Failed to decode OAuth response JSON. Error: @error', [
        '@error' => 'Decoded value is not an array.',
      ]);
      return;
    }

    if (empty($content['access_token'])) {
      $this->logger->error('access_token missing from OAuth response. Available keys: @keys. Available values: @values', [
        '@keys' => implode(', ', array_keys($content)),
        '@values' => implode(', ', array_map([$this, 'stringifyValueForLog'], $content)),
      ]);
      return;
    }

    try {
      $encodedPayload = Json::encode($content);
      $this->ensureFreshSession();
      $this->session->set('access_token', $encodedPayload);
      $response->setContent(Json::encode(['session' => $this->session->getId()]));
      $response->headers->set('Content-Type', 'application/json');
    }
    catch (\InvalidArgumentException $exception) {
      $this->logger->error('Failed to encode OAuth payload for session storage. Error: @error', [
        '@error' => $exception->getMessage(),
      ]);
      return;
    }
    catch (\Exception $e) {
      $this->logger->error('Exception in OAuth response subscriber: @message', [
        '@message' => $e->getMessage(),
      ]);
      return;
    }
  }

  /**
   * Stringifies a value for logging.
   */
  protected function stringifyValueForLog(mixed $value): string {
    if ($value === NULL) {
      return 'NULL';
    }
    if (is_scalar($value)) {
      return (string) $value;
    }

    try {
      return Json::encode($value);
    }
    catch (\InvalidArgumentException $exception) {
      return gettype($value);
    }
  }

  /**
   * Determines if a session wrapper was requested.
   */
  protected function sessionRequested(mixed $value): bool {
    return !in_array($value, [FALSE, '0', 0, NULL, ''], TRUE);
  }

  /**
   * Ensures a fresh session is available for storing the token payload.
   */
  protected function ensureFreshSession(): void {
    if ($this->session->isStarted()) {
      $this->session->migrate(TRUE);
      return;
    }

    $this->session->setId(Crypt::randomBytesBase64());
    $this->session->start();
  }

  /**
   * Determines if the matched controller handles OAuth token requests.
   */
  protected function isOauthTokenController(string $controller): bool {
    if ($controller === '' || !str_contains($controller, '::')) {
      return FALSE;
    }

    [$class, $method] = explode('::', $controller, 2);
    if (strcasecmp($method, 'token') !== 0 || !class_exists($class)) {
      return FALSE;
    }

    return is_a($class, Oauth2Token::class, TRUE);
  }

}
