<?php

namespace Drupal\tfa_headless\Service;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Utility\Crypt;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\encrypt\EncryptionProfileManagerInterface;
use Drupal\encrypt\EncryptServiceInterface;
use Drupal\rest\ResourceResponse;
use Drupal\tfa\TfaUserDataTrait;
use Drupal\user\UserDataInterface;
use Otp\Otp;
use ParagonIE\ConstantTime\Encoding;
use Psr\Log\LoggerInterface;

/**
 * The service for Headless TFA support.
 */
class TfaHeadlessService {
  use TfaUserDataTrait;

  /**
   * Un-encrypted seed.
   *
   * @var string
   */
  protected $seed;
  /**
   * The tfa config.
   *
   * @var mixed
   */
  protected $tfaConfig;
  /**
   * Encryption profile.
   *
   * @var \Drupal\encrypt\EncryptionProfileInterface
   */
  protected $encryptionProfile;
  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected $time;
  /**
   * The encrypt service.
   *
   * @var \Drupal\encrypt\EncryptServiceInterface
   */
  protected $encryptService;
  /**
   * The encryption profile manager.
   *
   * @var \Drupal\encrypt\EncryptionProfileManagerInterface
   */
  protected $encryptionProfileManager;
  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;
  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;
  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;
  /**
   * The session handler.
   *
   * @var \SessionHandlerInterface
   */
  protected $sessionHandler;

  /**
   * Logger channel.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new TfaHeadlessService object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    UserDataInterface $user_data,
    EncryptionProfileManagerInterface $encryption_profile_manager,
    EncryptServiceInterface $encrypt_service,
    TimeInterface $time,
    EntityTypeManagerInterface $entityTypeManager,
    \SessionHandlerInterface $session_handler,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->configFactory = $config_factory;
    $this->tfaConfig = $config_factory->get('tfa.settings')->get('validation_plugin_settings')['tfa_totp'];
    $this->userData = $user_data;
    $this->encryptionProfileManager = $encryption_profile_manager;
    $this->encryptService = $encrypt_service;
    $this->time = $time;
    $this->entityTypeManager = $entityTypeManager;
    $this->sessionHandler = $session_handler;
    $this->logger = $logger_factory->get('tfa_headless');
  }

  /**
   * Validate the TFA code.
   *
   * @param string $code
   *   The TFA code.
   * @param ?string $seed
   *   The TFA seed.
   *
   * @return bool
   *   Is code valid.
   */
  public function validate(string $code, $seed = NULL): bool {
    if ($this->currentUser === NULL) {
      return FALSE;
    }

    $this->getSeed();
    $code = preg_replace('/\s+/', '', $code);
    $current_window_base = floor((time() / 30)) - $this->tfaConfig['time_skew'];

    $otp = new Otp();
    $token_valid = (($seed ? $seed : $this->seed) && ($validated_window = $otp->checkHotpResync(
      Encoding::base32DecodeUpper($seed ? $seed : $this->seed),
      $current_window_base,
      $code,
      $this->tfaConfig['time_skew'] * 2
    )));
    if ($token_valid) {
      $this->setUserData(
        'tfa',
        ['tfa_totp_time_window' => $validated_window],
        $this->currentUser->id(), $this->userData
      );
    }
    return $token_valid;
  }

  /**
   * Get seed for this account.
   *
   * @return string
   *   Decrypted account OTP seed or FALSE if none exists.
   */
  public function getSeed(): ?string {
    if ($this->currentUser === NULL) {
      return NULL;
    }

    // Lookup seed for account and decrypt.
    $result = $this->getUserData('tfa', 'tfa_totp_seed', $this->currentUser->id(), $this->userData);

    if (!empty($result)) {
      $encrypted = base64_decode($result['seed']);
      $seed = $this->decrypt($encrypted);
      if (!empty($seed)) {
        $this->seed = $seed;
        return $seed;
      }
    }
    return NULL;
  }

  /**
   * Store validated code to prevent replay attack.
   *
   * @param string $code
   *   The validated code.
   */
  public function storeAcceptedCode(string $code): void {
    if ($this->currentUser === NULL) {
      return;
    }

    $code = preg_replace('/\s+/', '', $code);
    $hash = Crypt::hashBase64($code);

    // Store the hash made using the code in users_data.
    $store_data = ['tfa_accepted_code_' . $hash => $this->time->getRequestTime()];
    $this->setUserData('tfa', $store_data, $this->currentUser->id(), $this->userData);
  }

  /**
   * Save seed for account.
   *
   * @param string $seed
   *   Un-encrypted seed.
   */
  public function storeSeed(string $seed): void {
    if ($this->currentUser === NULL) {
      return;
    }

    // Encrypt seed for storage.
    $encrypted = $this->encrypt($seed);

    $record = [
      'tfa_totp_seed' => [
        'seed' => base64_encode($encrypted),
        'created' => $this->time->getRequestTime(),
      ],
    ];

    $this->setUserData('tfa', $record, $this->currentUser->id(), $this->userData);
  }

  /**
   * Encrypt a plaintext string.
   *
   * Should be used when writing codes to storage.
   *
   * @param string $data
   *   The string to be encrypted.
   *
   * @return string
   *   The encrypted string.
   *
   * @throws \Drupal\encrypt\Exception\EncryptException
   */
  public function encrypt(string $data): string {
    $encryptionProfileId = $this->configFactory->get('tfa.settings')->get('encryption');

    return $this->encryptService->encrypt(
      $data,
      $this->encryptionProfileManager->getEncryptionProfile($encryptionProfileId)
    );
  }

  /**
   * Decrypt a encrypted string.
   *
   * Should be used when reading codes from storage.
   *
   * @param string $data
   *   The string to be decrypted.
   *
   * @return string
   *   The decrypted string.
   *
   * @throws \Drupal\encrypt\Exception\EncryptionMethodCanNotDecryptException
   * @throws \Drupal\encrypt\Exception\EncryptException
   */
  public function decrypt(string $data): string {
    $encryptionProfileId = $this->configFactory->get('tfa.settings')->get('encryption');

    return $this->encryptService->decrypt(
      $data,
      $this->encryptionProfileManager->getEncryptionProfile($encryptionProfileId)
    );
  }

  /**
   * Getter for user data.
   *
   * @return \Drupal\user\UserData
   *   The user data.
   */
  public function getGlobalUserData() {
    return $this->userData;
  }

  /**
   * Get the session based on the session ID.
   *
   * @param string $session_id
   *   The session ID.
   */
  public function getSession($session_id) {
    if (!is_string($session_id) || $session_id === '') {
      return NULL;
    }

    try {
      $result = $this->sessionHandler->read($session_id);
    }
    catch (\Exception $exception) {
      return NULL;
    }

    if ($result) {
      $session_data = $this->decodeSessionData($result);
      $raw_token = $session_data['access_token']
        ?? ($session_data['_sf2_attributes']['access_token'] ?? NULL);

      if ($raw_token !== NULL) {
        $decoded = json_decode($raw_token, TRUE);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
          return $decoded;
        }
      }
    }

    return NULL;
  }

  /**
   * Decode the native PHP session payload into an array.
   *
   * @param string $session_data
   *   Raw session payload.
   */
  protected function decodeSessionData(string $session_data): array {
    $decoded = [];
    $offset = 0;
    $length = strlen($session_data);

    while ($offset < $length) {
      $separator = strpos($session_data, '|', $offset);
      if ($separator === FALSE) {
        break;
      }

      $key = substr($session_data, $offset, $separator - $offset);
      $offset = $separator + 1;
      $serialized_value = substr($session_data, $offset);

      $value = @unserialize($serialized_value, ['allowed_classes' => FALSE]);
      if ($value === FALSE && strncmp($serialized_value, 'b:0;', 4) !== 0) {
        break;
      }

      $decoded[$key] = $value;
      $offset += strlen(serialize($value));
    }

    return $decoded;
  }

  /**
   * Get the user based on the oauth token.
   *
   * @param string $oauth_token
   *   The oauth token.
   *
   * @return \Drupal\user\UserInterface|null
   *   The user entity if found, otherwise NULL.
   */
  public function getUser($oauth_token) {
    if (!is_string($oauth_token) || $oauth_token === '' || strpos($oauth_token, '.') === FALSE) {
      return NULL;
    }

    [$encoded_header] = explode('.', $oauth_token, 2);
    $header_json = base64_decode($encoded_header, TRUE);
    if ($header_json === FALSE) {
      return NULL;
    }

    $decoded_header = json_decode($header_json, TRUE);
    if (!is_array($decoded_header) || empty($decoded_header['jti'])) {
      return NULL;
    }

    $token_entity = $this->entityTypeManager
      ->getStorage('oauth2_token')
      ->loadByProperties(['value' => $decoded_header['jti']]);

    $token_entity = reset($token_entity);

    if ($token_entity) {
      $expire_timestamp = $token_entity->get('expire')->value;

      $now = time();

      if ($expire_timestamp > $now) {
        $this->currentUser = $token_entity->get('auth_user_id')->entity;
        return $this->currentUser;
      }
    }

    return NULL;
  }

  /**
   * Resolve session payload and authenticated user.
   */
  public function resolveSessionUser(string $session_id): ?array {
    $session = $this->getSession($session_id);
    if (!$session || empty($session['access_token'])) {
      $this->logger->warning('Session @session missing or lacks access token during TFA flow.', ['@session' => $session_id]);
      return NULL;
    }

    $user = $this->getUser($session['access_token']);
    if ($user === NULL) {
      $this->logger->warning('No user found for access token in session @session during TFA flow.', ['@session' => $session_id]);
      return NULL;
    }

    return [
      'session' => $session,
      'user' => $user,
    ];
  }

  /**
   * Build a cache-disabled REST response.
   */
  public function buildResponse($data, int $status = 200): ResourceResponse {
    $response = new ResourceResponse($data, $status);
    $cache = new CacheableMetadata();
    $cache->setCacheMaxAge(0);
    $response->addCacheableDependency($cache);
    return $response;
  }

  /**
   * Build a cache-disabled REST error response.
   */
  public function buildErrorResponse($data, int $status): ResourceResponse {
    return $this->buildResponse($data, $status);
  }

}
