<?php

namespace Drupal\theme_breakpoints_js;

use Drupal\breakpoint\BreakpointManagerInterface;
use Drupal\Core\Theme\ThemeInitializationInterface;
use Drupal\Core\Theme\ThemeManagerInterface;

/**
 * The ThemeBreakpointJs service for convenient loading of breakpoints.
 *
 * @package Drupal\theme_breakpoints_js
 */
class ThemeBreakpointsJs {

  /**
   * A list of currently known active theme objects.
   *
   * @var \Drupal\Core\Theme\ActiveTheme[]
   */
  protected array $activeThemes;

  /**
   * A list of loaded breakpoints, keyed by theme name.
   *
   * @var array
   */
  protected array $breakpointsByTheme;

  /**
   * ThemeBreakpointsJs constructor.
   *
   * @param \Drupal\breakpoint\BreakpointManagerInterface $breakpointManager
   *   The breakpoint manager.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $themeManager
   *   The theme manager.
   * @param \Drupal\Core\Theme\ThemeInitialization $themeInitialization
   *   The theme initialization service.
   */
  public function __construct(protected BreakpointManagerInterface $breakpointManager, protected ThemeManagerInterface $themeManager, protected ThemeInitializationInterface $themeInitialization) {
    $this->activeThemes = [];
    $this->breakpointsByTheme = [];
  }

  /**
   * Gets the defined breakpoints for the active theme of the current route.
   *
   * @return \Drupal\breakpoint\BreakpointInterface[]
   *   The breakpoints.
   */
  public function getBreakpointsForActiveTheme() {
    $theme = $this->themeManager->getActiveTheme();
    $this->activeThemes[$theme->getName()] = $theme;
    return $this->getBreakpoints($theme->getName());
  }

  /**
   * Returns defined breakpoints for the provided theme.
   *
   * When the given theme does not have defined any breakpoints by itself,
   * the base theme's breakpoint definitions will be loaded instead (if any).
   *
   * @param string $theme_name
   *   The machine name of the theme.
   *
   * @return \Drupal\breakpoint\BreakpointInterface[]
   *   The breakpoints, keyed by machine name without theme prefix.
   */
  public function getBreakpoints($theme_name) {
    if (!isset($this->breakpointsByTheme[$theme_name])) {
      $theme = !empty($this->activeThemes[$theme_name]) ? $this->activeThemes[$theme_name]
        : $this->themeInitialization->getActiveThemeByName($theme_name);
      $this->activeThemes[$theme_name] = $theme;
      $base_themes = $theme->getBaseThemeExtensions();
      $theme_candidates = !empty($base_themes) ? array_keys($base_themes) : [];
      array_unshift($theme_candidates, $theme_name);

      $this->breakpointsByTheme[$theme_name] = [];
      foreach ($theme_candidates as $candidate_name) {
        if (($breakpoints = $this->breakpointManager->getBreakpointsByGroup($candidate_name)) && !empty($breakpoints)) {
          foreach ($breakpoints as $id => $breakpoint) {
            $machine_name = preg_replace('/^' . $candidate_name . '\./', '', $id);
            $this->breakpointsByTheme[$theme_name][$machine_name] = $breakpoint;
          }
          break;
        }
      }
    }
    return $this->breakpointsByTheme[$theme_name];
  }

}
