<?php

namespace Drupal\theme_change\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for the Example add and edit forms.
 */
class ThemeChangeForm extends EntityForm {

  /**
   * The theme handler.
   *
   * @var \Drupal\Core\Extension\ThemeHandlerInterface
   */
  protected $themeHandler;

  /**
   * The route provider to load routes by name.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a ThemeChangeForm object.
   */
  public function __construct(ThemeHandlerInterface $themeHandler, RouteProviderInterface $route_provider, MessengerInterface $messenger) {
    $this->themeHandler = $themeHandler;
    $this->routeProvider = $route_provider;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('theme_handler'),
      $container->get('router.route_provider'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    $theme_change = $this->entity;
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $theme_change->label(),
      '#description' => $this->t("Label for the ThemeChange."),
      '#required' => TRUE,
    ];
    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $theme_change->id(),
      '#machine_name' => [
        'exists' => [$this, 'exist'],
      ],
      '#disabled' => !$theme_change->isNew(),
    ];
    $form['type'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Route/Path'),
      '#options' => [
        'route' => $this->t('Route'),
        'path' => $this->t('Path'),
      ],
      '#default_value' => $theme_change->getType(),
    ];
    $form['route'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter Route'),
      '#description' => $this->t("Enter route to change theme."),
      '#states' => [
        'visible' => ['select[name=type]' => ['value' => 'route']],
      ],
      '#default_value' => $theme_change->getRoute(),
    ];
    $form['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Enter Path'),
      '#description' => $this->t("Enter Path to change theme. And also Supports wildcards like(/user/*, /node/*)"),
      '#states' => [
        'visible' => ['select[name=type]' => ['value' => 'path']],
      ],
      '#default_value' => $theme_change->getPath(),
    ];
    $list_themes = [];
    $themes = $this->themeHandler->listInfo();
    foreach ($themes as $key => $value) {
      $list_themes[$key] = $this->themeHandler->getName($key);
    }
    $form['theme'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Theme'),
      '#options' => $list_themes,
      '#default_value' => $theme_change->getTheme(),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $route = $form_state->getValue('route');
    $path = $form_state->getValue('path');
    $type = $form_state->getValue('type');
    $route_provider = $this->routeProvider;
    $exists = count($route_provider->getRoutesByNames([$route])) === 1;
    // If no route exists.
    if ($type == 'route' && $route && !$exists) {
      $form_state->setErrorByName('route', $this->t('%route Route Doesnot exists', ['%route' => $route]));
    }
    // If type is route and path value is filled.
    if ($type == 'route' && !$route && $path) {
      $form_state->setErrorByName('type', $this->t("Select type as Path"));
    }
    // If type is path and route value is filled.
    if ($type == 'path' && !$path && $route) {
      $form_state->setErrorByName('type', $this->t("Select type as Route"));
    }
    if ($type == 'path' && $path[0] !== '/') {
      $form_state->setErrorByName('path', $this->t('The path needs to start with a slash.'));
    }
    // If route and path values are not filled.
    if (!$route && !$path) {
      $form_state->setErrorByName('route', $this->t('Route/Path Required'));
      $form_state->setErrorByName('path');
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $theme_change = $this->entity;
    $status = $theme_change->save();
    if ($status) {
      $this->messenger->addMessage($this->t('Saved the %label theme change.', ['%label' => $theme_change->label()]));
    }
    else {
      $this->messenger->addWarning($this->t('The %label theme change was not saved.', ['%label' => $theme_change->label()]));
    }
    $form_state->setRedirect('entity.theme_change.collection');
  }

  /**
   * Helper function to check whether ThemeChange configuration entity exists.
   */
  public function exist($id) {
    $storage = $this->entityTypeManager->getStorage('theme_change');
    $query = $storage->getQuery();
    $entity = $query
      ->condition('id', $id)
      ->accessCheck(FALSE)
      ->execute();
    return (bool) $entity;
  }

}
